// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.4
import AppStyle 1.0
import AppDialogs 1.0
import AppControls 1.0
import "constants.js" as Constants
import QtQuick.Layouts 1.12

Item {
    id: printView

    objectName: Constants.States.PRINTING
    //property real percent: printerModel.totalGCodeLine > 0
    //                       ? printerModel.currentGCodeLine / printerModel.totalGCodeLine
    //                       : 0

    ConfirmDialog {
        id: confirmCancelWindow
        visible: false
        z: 50
        anchors {
            verticalCenterOffset: -AppStyle.theme.headHeight * 0.5
            centerIn: parent
        }
        msg: qsTr("Cancel print?")
        onYes: {
            printerModel.cancelPrint()
        }
    }

    ConfirmDialog {
        id: confirmPauseWindow
        anchors {
            verticalCenterOffset: -AppStyle.theme.headHeight * 0.5
            centerIn: parent
        }
        visible: false
        z: 50
        msg: qsTr("Are you sure you want to pause the job, it can cause defects of the printing")
        onYes: {
            printerModel.pausePrint()
        }
    }

    Item {
        id: infoItem
        width: parent.width * 0.65
        anchors {
            top: parent.top
            left: parent.left
            bottom: parent.bottom
        }

        Column {
            spacing: AppStyle.theme.defaultMargin * 0.5
            width: parent.width
            anchors {
                verticalCenterOffset: 26
                centerIn: parent
            }

            Text {
                id: lineText
                width: parent.width
                horizontalAlignment: Text.AlignHCenter

                font {
                    family: AppStyle.theme.secondFontFamily
                    capitalization: Font.AllUppercase
                    pixelSize: 30
                }
                color: AppStyle.theme.textColor
                text: switch(coreModel.state) {
                      case Constants.States.PRINTING:
                          return qsTr("Printing...")
                      case Constants.States.LOCAL:
                          return qsTr("Printing...")
                      case Constants.States.DOWNLOADING:
                          return qsTr("Downloading...")
                      case Constants.States.PAUSED:
                          return qsTr("Paused")
                      case Constants.States.ERROR:
                          return qsTr("Error")
                      case Constants.States.CONNECTING:
                          return qsTr("Error")
                      default:
                          return qsTr("Finished")
                      }
            }

            AppProgressBar {
                id: printProgress
                value: printerModel.percent / 100
                width: parent.width
                height: printLocalBtn.height
            }

            Timer {
                interval: 5000; running: true; repeat: true
                onTriggered: timeLeftText.text = printerModel.printTimeLeftStr !== "" ? qsTr("Print time left: %1").arg(printerModel.printTimeLeftStr) : ""
            }

            Text {
                id: timeLeftText
                width: parent.width
                color: AppStyle.theme.textColor
                text: printerModel.printTimeLeftStr !== "" ? qsTr("Print time left: %1").arg(printerModel.printTimeLeftStr) : ""
                horizontalAlignment: Text.AlignHCenter
                font.pixelSize: 28
            }

            Text {
                id: jobOwnerText
                width: parent.width
                color: AppStyle.theme.textColor
                text: printerModel.jobOwner !== "" ? qsTr("Owner: %1").arg(printerModel.jobOwner) : ""
                horizontalAlignment: Text.AlignHCenter
                font.pixelSize: 28
            }
        }
    }

    Item {
        id: ctrlColumn
        anchors {
            top: parent.top
            right: parent.right
            left: infoItem.right
            bottom: parent.bottom
        }

        Column {
            id: tempColumn
            spacing: 5
            anchors {
                right: parent.right
                top: parent.top
            }
            width: parent.width * 0.9

            KeyValueItem {
                width: parent.width
                keyText: AppStyle.theme.bedTempText + ":"
                valueText: qsTr("%1/%2%3C").arg(printerModel.heatBedTemp).arg(printerModel.heatBedTargetTemp).arg(AppStyle.degreeSymbol)
            }

            KeyValueItem {
                width: parent.width
                keyText: AppStyle.theme.e0TempText + ":"
                valueText: qsTr("%1/%2%3C").arg(printerModel.firstExtruderTemp).arg(printerModel.firstExtruderTargetTemp).arg(AppStyle.degreeSymbol)
            }
        }

        Column {
            id: buttonsColumn
            spacing: printLocalBtn.height * 0.5
            width: tempColumn.width
            anchors {
                right: parent.right
                bottom: parent.bottom
            }

            TextButton {
                id: printLocalBtn
                width: parent.width
                text: switch(coreModel.state) {
                      case Constants.States.PRINTING:
                          return qsTr("Pause")
                      case Constants.States.PAUSED:
                          return qsTr("Resume")
                      default:
                          return qsTr("Pause")
                      }
                onClicked: {
                    if (coreModel.state === Constants.States.PRINTING) {
                        confirmPauseWindow.visible = true
                    } else if (coreModel.state === Constants.States.PAUSED) {
                        printerModel.resumePrint()
                    }
                }
            }

            TextButton {
                id: shutDownBtn
                width: parent.width
                text: qsTr("Cancel")
                onClicked: {
                    confirmCancelWindow.visible = true
                }
            }
        }
    }

    Component.onCompleted: {
        rootItem.clearHeadMsg()
        rootItem.showHead()
    }
}
