// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)


import QtQuick 2.7
import AppControls 1.0
import AppStyle 1.0
import AppDialogs 1.0
import AppViews 1.0

AbstactPage {
    id: zOffsetMenu
    property real zOffset: printerModel.zOffsetStep.toFixed(2)

    Component.onCompleted: {
        rootItem.hideHead()
    }

    ConfirmDialog {
        id: proceedConfirm
        z: 105
        anchors {
            verticalCenterOffset: AppStyle.theme.headHeight * 0.6
            centerIn: parent
        }
        visible: true
        titleText: qsTr("Warning")
        titleColor: AppStyle.theme.warningTitleBackgraound
        msg: qsTr("Printer will now home and move to X250 Y100.\nAre you sure you want to proceed?")
        yesText: qsTr("Proceed")
        noText: qsTr("Back")
        onYes: {
            printerModel.startZoffsetCalibration()
        }
        onNo: {
            rootItem.showHead()
            rootItem.back()
        }
    }

    Connections {
        target: printerModel
        function onOn_zOffset() {
            zoffsetColumn.visible = true
            footer.nextBtnVisible = true
        }
    }

    pageBody: Item {
        anchors.fill: parent

        Column {
            id: tempColumn
            spacing: 5
            anchors {
                right: parent.right
                top: parent.top
            }
            width: parent.width
            visible: zoffsetWaitColumn.visible
            KeyValueItem {
                width: parent.width
                keyText: qsTr("Print bed:")
                valueText: qsTr("%1/%2%3C").arg(printerModel.heatBedTemp).arg(printerModel.heatBedTargetTemp).arg(AppStyle.degreeSymbol)
            }

            KeyValueItem {
                width: parent.width
                keyText: qsTr("Printhead:")
                valueText: qsTr("%1/%2%3C").arg(printerModel.firstExtruderTemp).arg(printerModel.firstExtruderTargetTemp).arg(AppStyle.degreeSymbol)
            }
        }

        Item {
            anchors.fill: parent

            Column {
                id: zoffsetWaitColumn
                anchors.centerIn: parent
                width: parent.width
                visible: !zoffsetColumn.visible
                spacing: AppStyle.theme.defaultMargin

                Item {
                    width: zoffsetWaitColumn.width
                    height: waitIndicator.height
                    AppBusyIndicator {
                        id: waitIndicator
                        anchors.centerIn: parent
                    }
                }

                Text {
                    width: zoffsetWaitColumn.width
                    horizontalAlignment: Text.AlignHCenter
                    color: AppStyle.theme.textColor
                    text: qsTr("Preparing for calibration\nPlease wait...")
                    font {
                        bold: true
                        pixelSize: 25
                    }
                }
            }

            Column {
                id: zoffsetColumn
                anchors.centerIn: parent
                visible: false
                spacing: AppStyle.theme.defaultMargin * 0.5

                Text {
                    id: zOffsetLabel
                    width: zOffsetSpinBox.width
                    height: zOffsetLabel.paintedHeight
                    horizontalAlignment: Text.AlignHCenter
                    color: AppStyle.theme.textColor
                    text: qsTr("Calibration probe Z offset\nYou can use the up button to decrease and the down button\nto increase the gap between the nozzle and the build plate.\nMake sure that the nozzle tip slightly touches the build plate.\nPress Save when done.")
                    font {
                        bold: true
                        pixelSize: 25
                    }
                }

                AppVertSpinBox {
                    id: zOffsetSpinBox
                    value: printerModel.zOffset
                    onUpPressed: {
                        printerModel.moveZOneStepUp()
                    }
                    onDownPressed: {
                        printerModel.moveZOneStepDown()
                    }
                }
            }
        }
    }

    footer {
        nextBtnText: qsTr("Save")
        nextBtnVisible: false
        backBtnVisible: !proceedConfirm.visible

        onBackClicked: {
            printerModel.executeGcodeLine("G90")
            printerModel.executeGcodeLine("M104 S0")
            printerModel.executeGcodeLine("M140 S0")
            printerModel.disableResponseCallbacks()
            rootItem.showHead()
            rootItem.back()
        }

        onNextClicked: {
            printerModel.zOffset = zOffsetSpinBox.value.toFixed(2)
            printerModel.saveZOffset()
            printerModel.disableResponseCallbacks()
            rootItem.showHead()
            rootItem.hideSetting()
        }
    }
}
