// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.2
import Qt.labs.folderlistmodel 2.13

import AppStyle 1.0
import AppControls 1.0
import AppViews 1.0
import "constants.js" as Constants

ListPage {
    id: selectFileView
    objectName: Constants.States.LOCAL

    function formatBytes(bytes, decimals) {
        if(bytes === 0) return '0 Bytes';
        var k = 1024,
        dm = decimals <= 0 ? 0 : decimals || 2,
        sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'],
        i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
    }

    FolderListModel {
        id: folderListModel
        rootFolder: "/media"
        nameFilters: ["*.gcode"]
        showDirsFirst: true
    }

    property int subFoldersCount: 0

    listViewItem {
        model: folderListModel
        header: Rectangle {
            id: headDotDot
            width: listViewItem.width
            height: visible ? listViewItem.defaultItemHeight : 0
            color: headDotDotMouseArea.pressed ? AppStyle.foreground : "transparent"
            visible: subFoldersCount > 0
            border {
                color: AppStyle.foreground
                width: 1
            }

            Text {
                id: dotdotText
                anchors {
                    left: parent.left
                    leftMargin: AppStyle.defaultMargin * 0.6
                }
                height: listViewItem.defaultItemHeight
                font.pixelSize: listViewItem.defaultItemHeight * 0.45
                text: ".."
                color: AppStyle.textColor
                horizontalAlignment: Text.AlignLeft
                verticalAlignment: Text.AlignVCenter
                elide: Text.ElideRight
            }

            MouseArea {
                id: headDotDotMouseArea
                anchors.fill: parent
                onClicked: {
                    var folderStr = folderListModel.folder.toString()
                    folderListModel.folder = folderStr.split('/').slice( 0, -1 ).join('/');
                    subFoldersCount--
                    listViewItem.currentIndex = -1

                    if (folderListModel.folder.toString().split("//")[1] === folderListModel.rootFolder.toString()) {
                        console.log("return to parent dir")
                    }
                }
            }
        }

        delegate: Rectangle {
            id: fileListDelegate
            width: listViewItem.width
            height: listViewItem.defaultItemHeight
            color: (fileListDelegateMouseArea.pressed || fileListDelegate.ListView.isCurrentItem) ? AppStyle.foreground : "transparent"

            border {
                color: AppStyle.foreground
                width: 1
            }

            Row {
                id: fileListRow
                height: parent.height
                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                    rightMargin: 30
                    leftMargin: 30
                    verticalCenter: parent.verticalCenter
                }

                Text {
                    id: filenameText
                    height: fileListRow.height
                    width: fileListRow.width * 0.75
                    font.pixelSize: fileListRow.height * 0.35
                    text: fileName
                    color: AppStyle.textColor
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter
                    elide: Text.ElideRight
                }

                Text {
                    id: sizeText
                    width: fileListRow.width * 0.25
                    height: fileListRow.height
                    font.pixelSize: fileListRow.height * 0.35
                    text: fileIsDir ? qsTr("Folder") : formatBytes(fileSize)
                    color: AppStyle.textColor
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            MouseArea {
                id: fileListDelegateMouseArea
                anchors.fill: parent
                onClicked: {
                    if (fileIsDir) {
                        folderListModel.folder += "/" + fileName
                        subFoldersCount++
                        listViewItem.currentIndex = -1
                        return
                    }

                    listViewItem.currentIndex = index
                }
            }

            Behavior on height { PropertyAnimation { duration: 250 } }
        }
    }

    footer.nextBtnText: qsTr("Print")
    footer.nextBtnVisible: listViewItem.currentIndex !== -1

    footer.onBackClicked: {
        rootItem.clearHeadMsg()
        rootItem.back()
    }

    footer.onNextClicked: {
        var filePath = folderListModel.get(listViewItem.currentIndex, "filePath")
        console.log("sending file to print: " + filePath)
        filesModel.printFile(filePath)
    }

    Component.onCompleted: {
        print("Current folder: " + folderListModel.parentFolder)
        rootItem.setHeadMsg(qsTr("SD card"))
    }
}
