// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.4
import AppStyle 1.0
import AppDialogs 1.0
import "../constants.js" as Constants

/*
    Filament wizzard root item that manages wizzard pages
*/

Item {
    id: filamentWizardRoot

    enum WizardType {
        Load,
        Unload,
        RemovePrinthead
    }

    property int wizardType: FilamentWizard.WizardType.Load
    readonly property alias stackViewDepth: filamentWizardStack.depth

    function back() {
        filamentWizardStack.pop()
    }

    function pushView(qml, props) {
        props = props || {};
        filamentWizardStack.push(qml, props)
    }

    function cancelWizard() {
        cancelConfirm.visible = true
    }

    function startTestTimer() {
        testTimer.start()
    }

    ConfirmDialog {
        id: cancelConfirm
        z: 105
        anchors {
            verticalCenterOffset: -AppStyle.headHeight * 0.5
            centerIn: parent
        }
        visible: false
        msg: qsTr("Cancel filament load/unload wizard?")
        onYes: {
            //Cancel Wizard
            printerModel.executeGcodeLine('M98 P"finish.g"')
            printerModel.executeGcodeLine('M108')
            rootItem.back()
        }
    }

    StackView {
        id: filamentWizardStack
        visible: true
        anchors.fill: parent
    }

    signal gcodeResponse(string line, bool success)

    Timer {
        id: testTimer
        interval: 1500; running: false; repeat: false
        onTriggered: { gcodeResponse("ok", true); gcodeResponse("ok", true) }
    }

    onGcodeResponse: {
        if (success) {
            console.log("Success response: " + line)
            filamentWizardStack.currentItem.footer.visible = true
            return
        }
        var errorMsg = qsTr("Gcode error response '%1'").arg(line)
        rootItem.showMessageDialog(Constants.MessageTypes.ERROR, errorMsg, "OK")
        printerModel.executeGcodeLine('M98 P"finish.g"')
        printerModel.executeGcodeLine('M108')
        rootItem.hideSetting()
    }

    Component.onCompleted: {
        printerModel.localModeOn()
        printerModel.enableResponseCallbacks()
        printerModel.onOn_responseCallback.connect(gcodeResponse)
        filamentWizardStack.push("GlovesWarningView.qml")
    }

    Component.onDestruction: {
        printerModel.localModeOff()
        printerModel.disableResponseCallbacks()
        printerModel.executeGcodeLine('M108')
    }
}
