﻿// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
//
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.4
import AppStyle 1.0
import AppControls 1.0

Item {
    id: joystickView
    width: AppStyle.rootItemWidth
    height: AppStyle.rootItemHeight

    property int currentX: 0
    property int currentY: 0
    property int currentZ: 0
    property real currentStep: 100
    property real currentStepE: 100
    property real defaultBtnSize: 64
    property int tempStep: 5
    signal responseCallback(string line, bool success)


    Component.onCompleted: {
        rootItem.hideHead()
        printerModel.localModeOn()
        printerModel.executeGcodeLine("T0")
        printerModel.executeGcodeLine("M106 S0")
        printerModel.executeGcodeLine("M302 P1")
        printerModel.executeGcodeLine("G91")
        printerModel.executeGcodeLine("M83")
    }

    Row {
        spacing: 10
        anchors.centerIn: parent
        Column {
            spacing: 10
            Row {
                id: stepRow
                spacing: 25

                Text {
                    id: stepText
                    text: qsTr("Step mm")
                    color: AppStyle.textColor
                    font.pixelSize: 20
                    wrapMode: Text.WordWrap
                    width: 40
                    height: 40
                }

                ButtonGroup {
                    id: tempBtnGrp
                    buttons: stepTempRow.children
                }

                Row {
                    id: stepTempRow
                    spacing: -1
                    Repeater {
                        id: stepRepeater
                        model: ["0.1", "1", "10", "20"]

                        AppRadioButton {
                            width: 52
                            height: 52
                            text: modelData
                            checked: index === 0
                            //                            onClicked: {
                            //                                currentStepField.text = modelData
                            //                            }
                        }
                    }
                }

                //                Column {
                //                    AppNumberField {
                //                        id: currentStepField
                //                        text: "100"
                //                        onValidated: {
                //                            currentStep = text
                //                        }
                //                    }
                //                    Text {
                //                        color: AppStyle.textColor
                //                        text: qsTr("Custom")
                //                    }
                //                }
            }

            JoystickControl {
                id: joystickCtrl
                property string tempStep: tempBtnGrp.checkedButton.text
                onXClicked: {

                    if (positive) {
                        printerModel.executeGcodeLine("G0 X" + tempStep)
                        console.log("+X Clicked val: " + tempStep)
                        return
                    }
                    printerModel.executeGcodeLine("G0 X-" + tempStep)
                    console.log("-X Clicked val: " + (-tempStep))
                }

                onYClicked: {
                    if (positive) {
                        printerModel.executeGcodeLine("G0 Y" + tempStep)
                        console.log("+Y Clicked val: " + tempStep)
                        return
                    }

                    printerModel.executeGcodeLine("G0 Y-" + tempStep)
                    console.log("-X Clicked val: " + (-tempStep))
                }

                onZClicked: {
                    if (positive) {
                        printerModel.executeGcodeLine("G0 Z" + tempStep)
                        console.log("+Z Clicked val: " + tempStep)
                        return
                    }

                    printerModel.executeGcodeLine("G0 Z-" + tempStep)
                    console.log("-Z Clicked val: " + (-tempStep))
                }
            }

            Column {
                Text {
                    color: AppStyle.textColor
                    width: homeBtnRow.width
                    font.pixelSize: 20
                    text: qsTr("Homing")
                    horizontalAlignment: Text.AlignRight
                }
                Row {
                    id: homeBtnRow
                    spacing: -1
                    Repeater {
                        model: ["X", "Y", "Z", "All"]
                        TextButton {
                            width: joystickView.defaultBtnSize
                            height: joystickView.defaultBtnSize
                            text: modelData
                            onClicked: {
                                switch(modelData) {
                                case "X":
                                    printerModel.executeGcodeLine("G28 X")
                                    console.log("X Home Pressed")
                                    return
                                case "Y":
                                    printerModel.executeGcodeLine("G28 Y")
                                    console.log("Y Home Pressed")
                                    return
                                case "Z":
                                    printerModel.executeGcodeLine("G28 Z")
                                    console.log("Z Home Pressed")
                                    return
                                case "All":
                                    printerModel.executeGcodeLine("G28")
                                    console.log("All Home Pressed")
                                    return
                                }
                            }
                        }
                    }
                }
            }
        }

        Rectangle {
            id: delimeterRect
            width: 1
            height: joystickView.height * 0.95
            color: AppStyle.foreground
            opacity: 0.5
        }

        Column {
            spacing: 10

            Grid {
                id: tempGrid
                rows: 4
                columns: 2

                rowSpacing: 10
                columnSpacing: 10

                Text {
                    id: bedTempText
                    font.pixelSize: 20
                    width: bedColumn.width
                    horizontalAlignment: Text.AlignHCenter
                    color: AppStyle.textColor
                    text: qsTr("Bed:\t%1/%2%3C").arg(printerModel.heatBedTemp).arg(printerModel.heatBedTargetTemp).arg(AppStyle.degreeSymbol)
                }

                Text {
                    id: extrTempText
                    font.pixelSize: 20
                    width: bedColumn.width
                    horizontalAlignment: Text.AlignHCenter
                    color: AppStyle.textColor
                    text: qsTr("Head:\t%1/%2%3C").arg(printerModel.firstExtruderTemp).arg(printerModel.firstExtruderTargetTemp).arg(AppStyle.degreeSymbol)
                }

                Column {
                    id: bedColumn
                    //Text {
                        //color: AppStyle.textColor
                        //width: parent.width
                        //horizontalAlignment: Text.AlignHCenter
                        //font.pixelSize: bedTempSpin.height * 0.3
                        //text: "Bed temp."
                    //}
                    AppSpinBox {
                        id: bedTempSpin
                        minValue: 0
                        maxValue: 120
                        unitText: "C"
                        stepSize: 5
                        value: printerModel.heatBedTemp
                        onValidated: {
                            printerModel.executeGcodeLine("M140 S" + value)
                            console.log("Bed temp New value " + value)
                        }
                    }
                }

                Column {
                    id: extruderTempColumn

                    //Text {
                        //color: AppStyle.textColor
                        //width: parent.width
                        //horizontalAlignment: Text.AlignHCenter
                        //font.pixelSize: bedTempSpin.height * 0.3
                        //text: "Printhead temp."
                    //}
                    AppSpinBox {
                        id: extruderTemp
                        minValue: 0
                        maxValue: 310
                        unitText: "C"
                        stepSize: 10
                        value: printerModel.firstExtruderTemp
                        onValidated: {
                            printerModel.executeGcodeLine("M104 S" + value)
                            console.log("Extruder temp " + value)
                        }

                    }
                }

                Column {
                    id: feedRateColumn
                    Text {
                        color: AppStyle.textColor
                        width: parent.width
                        horizontalAlignment: Text.AlignHCenter
                        font.pixelSize: bedTempSpin.height * 0.3
                        //text: "Extruder speed"
                        text: "Feed rate"
                    }
                    AppSpinBox {
                        id: feedRate
                        stepSize: 200
                        minValue: 200
                        maxValue: 3000
                        value: printerModel.joystickFeedrate
                        onValueChanged: {
                            validate()
                            printerModel.executeGcodeLine("G0 F" + value)
                            console.log("Feed rate " + value)
                        }
                    }
                }

                Column {
                    id: fanColum
                    Text {
                        color: AppStyle.textColor
                        width: parent.width
                        horizontalAlignment: Text.AlignHCenter
                        font.pixelSize: bedTempSpin.height * 0.3
                        text: "Fan"
                    }
                    AppSpinBox {
                        id: fanSpin

                        // TODO: check
                        minValue: 0
                        maxValue: 100
                        stepSize: 5
                        unitText: "%"
                        // TODO: set value from model
                        value: 0
                        onValueChanged: {

                            //printerModel.executeGcodeLine("M106 S" + Math.ceil(value * 2.5)); //FIXME 0 to 100% is 0 to 255 of S value
                            var fanValue, maxFan = 250
                            fanValue = Math.round(maxFan * value / 100)
                            printerModel.executeGcodeLine("M106 S" + fanValue); //FIXME 0 to 100% is 0 to 255 of S value
                            console.log("Fan speed " + fanValue);
                        }
                    }
                }

                Item {
                    width: fanSpin.width
                    height: fanColum.height
                    TextButton {
                        id: coolAllBtn
                        text: qsTr("Cool all")
                        textPixelSize: height * 0.35
                        width: joystickView.defaultBtnSize
                        height: joystickView.defaultBtnSize
                        anchors.centerIn: parent
                        onClicked: {
                            printerModel.executeGcodeLine("M104 T0 S0\nM140 S0") //TODO check for second extruder and add M104 T1 if needed
                            console.log("Cool all")
                        }
                    }
                }

                Item {
                    width: fanSpin.width
                    height: fanColum.height
                    TextButton {
                        id: stopFanBtn
                        text: qsTr("Stop fan")
                        textPixelSize: height * 0.35
                        width: joystickView.defaultBtnSize
                        height: joystickView.defaultBtnSize
                        anchors.centerIn: parent
                        onClicked: {
                            printerModel.executeGcodeLine("M107") //TODO check for second extruder and add M104 T1 if needed
                            console.log("Stop Fan")
                        }
                    }
                }
            }

            Item {
                width: tempGrid.width
                height: eColumn.height
                Column {
                    id: eColumn
                    anchors.centerIn: parent
                    Item {
                        id: eBtnLabels
                        width: eRow.width
                        height: retractText.height
                        Text {
                            id: retractText
                            anchors.left: parent.left
                            width: retractBtn.width
                            color: AppStyle.textColor
                            horizontalAlignment: Text.AlignHCenter
                            font.pixelSize: retractBtn.height * 0.3
                            text: qsTr("Retract")
                        }

                        Text {
                            anchors.right: parent.right
                            width: extrudeBtn.width
                            color: AppStyle.textColor
                            horizontalAlignment: Text.AlignHCenter
                            font.pixelSize: retractBtn.height * 0.3
                            text: qsTr("Extrude")
                        }
                    }



                    Row {
                        id: eRow
                        spacing: 5
                        property string eStepText: eStepGrp.checkedButton.text

                        TextButton {
                            id: retractBtn
                            text: qsTr("-E")
                            textPixelSize: height * 0.5
                            width: joystickView.defaultBtnSize
                            height: joystickView.defaultBtnSize

                            onClicked: {
                                printerModel.executeGcodeLine("G0 E-" + eRow.eStepText) //TODO check for second extruder and add M104 T1 if needed
                                console.log("Retract pressed e step: -" + eRow.eStepText)
                            }
                        }

                        ButtonGroup {
                            id: eStepGrp
                            buttons: eStepRow.children
                        }

                        Row {
                            id: eStepRow
                            topPadding: 5
                            spacing: -1
                            Repeater {
                                model: ["0.1", "1", "10", "20"]
                                AppRadioButton {
                                    checked: index === 0
                                    width: 52
                                    height: 52
                                    text: modelData
                                }
                            }
                        }

                        //                        Item {
                        //                            height: eRow.height
                        //                            width: eStepField.width

                        //                            AppNumberField {
                        //                                id: eStepField
                        //                                anchors.centerIn: parent
                        //                                text: "100"
                        //                                onValidated: {
                        //                                    currentStepE = text
                        //                                }
                        //                            }

                        //                            Text {
                        //                                anchors.top: eStepField.bottom
                        //                                color: AppStyle.textColor
                        //                                text: qsTr("Custom")
                        //                            }
                        //                        }

                        TextButton {
                            id: extrudeBtn
                            text: qsTr("+E")
                            textPixelSize: height * 0.5
                            width: joystickView.defaultBtnSize
                            height: joystickView.defaultBtnSize

                            onClicked: {
                                printerModel.executeGcodeLine("G0 E" + eRow.eStepText) //TODO check for second extruder and add M104 T1 if needed
                                console.log("Extrude pressed e step: " + eRow.eStepText)
                            }
                        }
                    }
                }
            }
        }
    }

    TextButton {
        id: backBtn
        text: qsTr("Back")
        anchors {
            bottom: parent.bottom
            left: parent.left
        }
        height: 55

        onClicked: {
            printerModel.executeGcodeLine("G90")
            printerModel.executeGcodeLine("M302 P0")
            printerModel.localModeOff()
            rootItem.back()
            rootItem.showHead()
        }
    }
}
