// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.2

import AppStyle 1.0
import AppControls 1.0
import AppViews 1.0

import "constants.js" as Constants

ListPage {
    id: selectWifiView

    property bool showForget: false
    property var demoModel: [{
            "ssid": "Surprising Black Big Long Great Wifi SSID 1",
            "signalLevel": 1,
            "secured": false,
            "connected": true
        }, {
            "ssid": "Test1",
            "signalLevel": 1,
            "secured": false,
            "connected": false
        }, {
            "ssid": "Test2",
            "signalLevel": 2,
            "secured": false,
            "connected": false
        }, {
            "ssid": "Test23",
            "signalLevel": 4,
            "secured": true,
            "connected": false
        }, {
            "ssid": "Surprising Black Big Long Great Wifi SSID 2",
            "signalLevel": 4,
            "secured": true,
            "connected": false
        }, {
            "ssid": "Test4",
            "signalLevel": 2,
            "secured": true,
            "connected": false
        }]



    Timer {
        id: refreshScanTimer
        running: false
        repeat: false
        interval: 5000
        onTriggered: {
            networkModel.scanWifi()
        }
    }

    BusyLayer {
        id: listBusyLayer
        color: "transparent"
        anchors.fill: parent
    }

    listViewItem {
//        model: demoModel
        model: networkModel.wifiStationList
        delegate: Rectangle {
            id: wifiDelegate
            width: listViewItem.width
            property real defaultHeight: 64
            height: visible ? (showForget && modelData.connected ? defaultHeight * 2 : defaultHeight) : 0
            color: wifiDelegateMouseArea.pressed ? AppStyle.foreground : "transparent"
            visible: modelData.ssid !== ""
            border {
                color: AppStyle.foreground
                width: 1
            }

            Row {
                height: wifiDelegate.defaultHeight
                anchors {
                    top: parent.top
                    left: parent.left
                    rightMargin: 30
                    verticalCenter: parent.verticalCenter
                }

                Image {
                    id: wifiListIcon
                    asynchronous: true
                    source: switch(modelData.signalLevel) {
                            case 0:
                                return "img/wifi-icon-0.png"
                            case 1:
                                return "img/wifi-icon-1.png"
                            case 2:
                                return "img/wifi-icon-2.png"
                            case 3:
                                return "img/wifi-icon-3.png"
                            case 4:
                                return "img/wifi-icon-4.png"
                            }

                    sourceSize.width: wifiDelegate.defaultHeight
                    sourceSize.height: wifiDelegate.defaultHeight
                }

                Item {
                    width: wifiDelegate.defaultHeight
                    height: wifiDelegate.defaultHeight

                    Image {
                        visible: modelData.secured
                        asynchronous: true
                        source: "img/lock-icon.png"
                        sourceSize.width: wifiDelegate.defaultHeight
                        sourceSize.height: wifiDelegate.defaultHeight
                    }
                }

                Text {
                    height: wifiDelegate.defaultHeight
                    width: wifiDelegate.width * 0.5
                    font.pixelSize: wifiDelegate.defaultHeight * 0.35
                    text: modelData.ssid
                    clip: true
                    color: "white"
                    verticalAlignment: Text.AlignVCenter
                    elide: Text.ElideRight
                }
            }

            MouseArea {
                id: wifiDelegateMouseArea
                anchors.fill: parent
                onClicked: {
                    if (modelData.connected) {
                        // showForget = !showForget
                        return
                    }

                    rootItem.pushView("WifiPasswdView.qml", { "ssid" : modelData.ssid, "secured" : modelData.secured })
                }
            }

            Column {
                id: btnColumn
                anchors {
                    right: parent.right
                    rightMargin: wifiDelegate.defaultHeight * 0.4
                    verticalCenter: parent.verticalCenter
                }

                spacing: wifiDelegate.height * 0.1

                TextButton {
                    id: disconnectBtn
                    height: wifiDelegate.defaultHeight * 0.7
                    width: 177

                    text: qsTr("Disconnect")
                    visible: modelData.connected
                    onClicked: {
                        listBusyLayer.show()
                        networkModel.disconnectWifi()
                    }
                }

                TextButton {
                    id: forgetBtn
                    width: 177
                    height: visible ? wifiDelegate.defaultHeight * 0.7 : 0
                    text: qsTr("Forget")
                    backgroundColor: AppStyle.red
                    visible: showForget && modelData.connected
                    onClicked: {
                        console.log("TODO: forget wifi slot in python model")
                    }
                }
            }

            Behavior on height { PropertyAnimation { duration: 250 } }
        }
    }

    footer {
        textInfo: listViewItem.model.length === 0 ? qsTr("Loading...") : qsTr("Select network")
        nextBtnText: qsTr("Refresh")
        onNextClicked: {
            listBusyLayer.show()
            networkModel.scanWifi()
        }
        onBackClicked: {
            rootItem.clearHeadMsg()
            rootItem.back()
        }
    }

    Connections {
        target: networkModel
        onOn_wifiListChanged: {
            listBusyLayer.hide()
        }
        onOn_networkConnectChanged: {
            listBusyLayer.hide()
        }
    }

    Component.onCompleted: {
        rootItem.setHeadMsg(qsTr("Wi-Fi scan results"))
        networkModel.scanWifi()
        refreshScanTimer.start()
    }
}
