# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
# Built in San Francisco.

# This software is distributed under a commercial license for personal,
# educational, corporate or any other use.
# The software as a whole or any parts of it is prohibited for distribution or
# use without obtaining a license from 3D Control Systems, Inc.

# All software licenses are subject to the 3DPrinterOS terms of use
# (available at https://www.3dprinteros.com/terms-and-conditions/),
# and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import http.client
import logging
import os
import threading
import time
import paths

import config
import filename
from base_sender import BaseSender
from makerbotapi import Makerbot
from makerbotapi import MakerBotError
from m5th_cam import M5thCamera


class BirdwingNetworkConnection(Makerbot):

    def __init__(self, ip, auth_code=None, verbose=False, auth_timeout=120, logger=None):
        self.logger = logger
        Makerbot.__init__(self, ip, auth_code)
        self.debug_http = verbose
        self.debug_jsonrpc = verbose
        self.debug_fcgi = verbose
        self.auth_timeout = auth_timeout

    def _debug_print(self, protocol, direction, content):
        if self.logger:
            self.logger.debug("(%s) %s: %s" % (protocol, direction, content))
        else:
            Makerbot._debug_print(self, protocol, direction, content)

    def put_file(self, gcode_file, remote_path):
        token = self.get_access_token(context='put')
        if not token:
            raise Exception("No access token for PUT file")
        put_token = str(token)
        host = "%s:%i" % (self.host, 80)
        con = http.client.HTTPConnection(host)
        remote_path = "%s?token=%s" % (remote_path, put_token)
        if self.debug_http:
            self._debug_print('HTTP', 'REQUEST', "PUT FILE:  %s%s" % (host, remote_path))
        con.request("PUT", remote_path, gcode_file)
        resp = con.getresponse()
        status_code = resp.status
        if self.debug_http:
            self._debug_print('HTTP', 'RESPONSE', "PUT FILE: status code: %s" % status_code)
        msg = resp.read()
        if self.debug_http:
            self._debug_print('HTTP', 'RESPONSE', 'Received http code %d, message\n%s' % (status_code, msg))
            # raise httplib.HTTPException(errcode)
        return status_code == 200

    def start_printing_file(self, file):
        if self.rpc_request_response_with_try("print", {"filepath": os.path.basename(file)}):
            return self.rpc_request_response_with_try("process_method", {"params": {}, "method": "build_plate_cleared"})

    def cancel_print(self):
        return self.rpc_request_response_with_try("cancel", {})

    def pause_print(self):
        return self.rpc_request_response_with_try("process_method", {"params": {}, "method": "suspend"})

    def unpause_print(self):
        return self.rpc_request_response_with_try("process_method", {"params": {}, "method": "resume"})

    def rpc_request_response_with_try(self, method, params, timeout=15):
        try:
            return self.rpc_request_response(method, params, timeout)
        except AssertionError:
            return

    def authenticate_to_printer(self):
        try:
            self.authenticate_json_rpc()
            return True
        except:
            pass


class Sender(BaseSender):

    REMOTE_URL_PATH = "/current_thing/"
    DEFAULT_FILE_NAME = "3dprinteros"
    DEFAULT_FILE_EXTENSION = ".makerbot"
    STEP_PRINTING_NAME = "printing"

    def __init__(self, parent, usb_info, profile):
        BaseSender.__init__(self, parent, usb_info, profile)
        self.logger = logging.getLogger(self.__class__.__name__)
        self.operational_flag = False
        self.printing_flag = False
        self.printing_started_flag = False
        self.pairing_needed = False
        self.percent = 0
        self.ip = usb_info.get('IP')
        self.available_methods = []
        self.serial_number = usb_info['SNR']
        self.timeout = profile["operational_timeout"]
        self.makerbot = None
        self.camera = None
        self.camera_thread = None
        self.verbose = config.get_settings()['verbose']
        if not self.ip:
            self.auth_and_monitoring_thread = None
            return
        self.path_to_file_with_auth_code = os.path.join(
            paths.CURRENT_SETTINGS_FOLDER,
            "%s_%s.auth_code" % (self.ip.replace(".", "-"), self.serial_number)
        )
        self.auth_and_monitoring_thread = threading.Thread(target=self.auth_and_monitoring)
        self.auth_and_monitoring_thread.start()
        #  TODO: Develop report to server info about this printer (in botstate)

    def auth_and_monitoring(self):
        self.logger.info("Connecting to BirdWing printer...")
        retries = 20
        while True:
            try:
                self.makerbot = BirdwingNetworkConnection(
                    self.ip, self.read_auth_code(), verbose=self.verbose, auth_timeout=self.timeout,
                    logger=self.logger)
                self.makerbot.connect()
                break
            except Exception as e:
                retries -= 1
                self.logger.info('Printer connecting retries left: %s' % retries)
                self.kill_makerbot_object()
                if self.stop_flag:
                    return
                is_blocking = retries <= 0
                message = "Cannot connect to printer, error message: %s" % str(e)
                self.parent.register_error(708, message, is_blocking=is_blocking)
                if is_blocking:
                    return
                time.sleep(5)
        if not self.makerbot.authenticate_to_printer():
            self.pairing_needed = True
            self.logger.debug("Press the flashing action button on your printer now")
            try:
                self.makerbot.authenticate_fcgi()
            except Exception as e:
                self.kill_makerbot_object()
                message = "Birdwing module failed in pairing your printer, error message: %s" % str(e)
                self.parent.register_error(707, message, is_blocking=True)
                return
            self.logger.debug("Authenticated with code: %s" % self.makerbot.auth_code)
            self.write_auth_code(self.makerbot.auth_code)
            if not self.makerbot.authenticate_to_printer():
                self.kill_makerbot_object()
                message = "Birdwing module can't authenticate printer after pairing."
                self.parent.register_error(706, message, is_blocking=True)
                return
        self.pairing_needed = False
        self.logger.info("...connected!")
        assert_retries_cnt = 0
        while not self.stop_flag:
            if self.parent.current_camera == 'Disable camera':
                self.stop_camera()
            else:
                self.start_camera()
            printer_state = None
            try:
                printer_state = self.makerbot.get_system_information()
            except AssertionError:
                assert_retries_cnt += 1
                if assert_retries_cnt < 10:
                    printer_state = True
                self.logger.info("Empty response (timeout, retry: %d)" % assert_retries_cnt)
                time.sleep(1)
                continue
            except Exception as e:
                self.parent.register_error(700, "Crash BirdWing module, exception: %s" % e, is_blocking=True)
                break
            finally:
                self.operational_flag = bool(printer_state)
            assert_retries_cnt = 0
            if printer_state:
                # print self.makerbot.firmware_version
                # print self.makerbot.iserial
                # print self.makerbot.machine_name
                # print self.makerbot.machine_type
                # print self.makerbot.vid
                # print self.makerbot.pid
                # print self.makerbot.bot_type
                toolhead = printer_state.toolheads[0]
                self.temps[1] = toolhead.current_temperature
                self.target_temps[1] = toolhead.target_temperature
                process = printer_state.current_process
                if process:
                    self.available_methods = process.methods
                    self.printing_flag = (process.step != 'completed')
                    if process.step == 'error_step' or process.step == 'failed':
                        self.logger.warning('error_step: %s' % process)
                        if process.error:
                            error = process.error
                            if error.get('code') == 1052:
                                error['description'] = 'Model has no raft. Please confirm from printer'
                            elif error.get('code') == 1049:
                                error['description'] = 'Print machine mismatch'
                        else:
                            error = {}
                        self.parent.register_error(
                            608, "Error step: %s" % error, is_info=True)
                    if process.step == self.STEP_PRINTING_NAME and process.progress:
                        # Dirty hack, because this printer can answer on first step "printing" 98%!
                        if self.percent != 0 or process.progress < 50:
                            self.percent = process.progress
                else:
                    if self.printing_flag and self.printing_started_flag and self.percent < 95:
                        self.parent.register_error(607, "Cancelled manually", is_blocking=True)
                    self.printing_flag = False
            time.sleep(0.5)
        self.kill_makerbot_object()

    def kill_makerbot_object(self):
        if self.makerbot:
            self.makerbot.disconnect_json_rpc()
            del self.makerbot
            self.makerbot = None

    def start_camera(self):
        if not self.camera:
            self.camera = M5thCamera(self)
        if self.camera_thread:
            if self.camera_thread.isAlive():
                return
            del self.camera_thread
        self.camera.stop_flag = False
        self.camera_thread = threading.Thread(target=self.camera.main_loop)
        self.camera_thread.start()

    def stop_camera(self):
        if self.camera_thread and self.camera_thread.isAlive():
            self.camera.stop_flag = True

    def load_gcodes(self, gcode_file):
        if not self.printing_flag:
            self.logger.info("Filename: %s" % self.filename)
            if self.filename:
                self.filename = filename.get_filename_ascii(self.filename, '?')
            if not self.filename:
                self.filename = self.DEFAULT_FILE_NAME
            self.logger.info("Filename after fixes: %s" % self.filename)
            path_file = self.REMOTE_URL_PATH + self.filename + self.DEFAULT_FILE_EXTENSION
            if not self.makerbot.put_file(gcode_file, path_file):
                self.parent.register_error(701, "Cannot upload file to printer", is_blocking=True)
                return
            if not self.makerbot.start_printing_file(path_file):
                self.parent.register_error(702, "Cannot start print", is_blocking=True)
                return
            self.percent = 0
            self.printing_started_flag = True
            self.logger.info("Success start print")
        else:
            return False

    def is_printing(self):
        return self.printing_flag

    def is_paused(self):
        return self.pause_flag

    def is_operational(self):
        if not self.ip:
            msg = "Makerbot 5th Generation is only supported over WiFi/Ethernet connection, not USB."
            self.parent.forced_state = 'error'
            self.parent.register_error(709, msg, static=True)
            return False
        return self.operational_flag

    def pause(self):
        # FIXME: Its ugly, need move to BirdwingConnector or better MakerBotAPI
        if not self.pause_flag and "suspend" in self.available_methods:
            self.makerbot.pause_print()
            self.pause_flag = True
        else:
            self.parent.register_error(703, "For now, cannot use command pause.", is_blocking=False)
            return False

    def unpause(self):
        # FIXME: Its ugly, need move to BirdwingConnector or better MakerBotAPI
        if self.pause_flag and "resume" in self.available_methods:
            self.makerbot.unpause_print()
            self.pause_flag = False
        else:
            self.parent.register_error(704, "For now, cannot use command resume.", is_blocking=False)
            return False

    def cancel(self):
        # Protection from some problems with cancel on Replicator+
        # First cancel doesn't not work if we have step: failed
        for i in range(1, 4):
            if self.printing_flag:
                self.logger.info("Cancel print #"+str(i))
                try:
                    self.makerbot.cancel_print()
                except MakerBotError as e:
                    if 'RPC Error code=-32000 ' in str(e):
                        break
                    raise e
                time.sleep(0.2)
        return False

    def get_percent(self):
        return self.percent

    def set_total_gcodes(self, length):
        pass

    def get_current_line_number(self):
        return 0

    def unbuffered_gcodes(self, gcodes):
        # TODO: Add report for not support this
        self.parent.register_error(705, "This printer don't support Gcodes", is_blocking=False)
        return None

    def close(self):
        self.logger.info("Closing BirdWingSender module...")
        self.stop_flag = True
        if self.auth_and_monitoring_thread and self.auth_and_monitoring_thread.is_alive():
            self.auth_and_monitoring_thread.join()
        self.logger.info("...closed")

    def read_auth_code(self):
        try:
            self.logger.info("Birdwing reading auth code from file: %s" % self.path_to_file_with_auth_code)
            file = open(self.path_to_file_with_auth_code, "r")
            auth_code = file.read()
            file.close()
            self.logger.info("Birdwing read auth code: %s" % auth_code)
            return auth_code
        except Exception as e:
            self.logger.info(
                "Cannot read from file: %s, have exception: %s" % (self.path_to_file_with_auth_code, str(e)))

    def write_auth_code(self, auth_code):
        try:
            self.logger.info(
                "Birdwing writing auth code: %s to file: %s" % (auth_code, self.path_to_file_with_auth_code))
            file = open(self.path_to_file_with_auth_code, "w")
            file.write(auth_code)
            file.close()
        except Exception as e:
            self.logger.info(
                "Cannot write to file: %s, have exception: %s" % (self.path_to_file_with_auth_code, str(e)))
