// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Window 2.2
import QtQuick.Controls 2.4
import AppStyle 1.0
import AppDialogs 1.0
import AppControls 1.0
import Keyboard 1.0
import "constants.js" as Constants

Item {

    id: rootItem
    visible: true
    width: AppStyle.rootItemWidth
    height: AppStyle.rootItemHeight

    property bool settingsOpened: false
    property bool isUpdating: false
    property var msgQueue: []
    property alias keyboardItem: keyboard
    readonly property bool networkConfirmVisible: !networkModel.networkConnected && !coreModel.offlineMode && !rootItem.settingsOpened

    function back() {
        stack.pop()
    }

    function pushView(qml, props) {
        props = props || {};
        stack.push(qml, props)
    }

    function showBusy() {
        busyLayer.visible = true
    }

    function hideBusy() {
        busyLayer.visible = false
    }

    function showHead() {
        appHead.visible = true
    }

    function hideHead() {
        appHead.visible = false
    }

    function keyboardVisible() {
        return keyboard.visible
    }

    function showKeyboard() {
        keyboard.source = "keyboard_us.xml"
        keyboard.visible = true
    }

    function showKeyboardNum() {
        keyboard.source = "keyboard_calc.xml"
        keyboard.visible = true
    }

    function hideKeyboard() {
        keyboard.visible = false
    }

    function setHeadMsg(msg) {
        appHead.headMsg = msg
    }

    function clearHeadMsg() {
        appHead.headMsg = ""
    }

    function showSetting() {
        if (!rootItem.settingsOpened) {
            rootItem.setHeadMsg("Settings")
            rootItem.settingsOpened = true
            rootItem.pushView("SettingsView.qml")
        }
        if (coreModel.state === Constants.States.CONNECTING) {
            hideBusy()
        }
    }

    function hideSetting() {
        rootItem.settingsOpened = false
        stack.pop(null)
        clearHeadMsg()
        checkState()
        rootItem.showHead()
        if (coreModel.state === Constants.States.CONNECTING) {
            showBusy()
        }
    }

    function hideSettingWithBusy() {
        rootItem.settingsOpened = false
        stack.pop(null)
        clearHeadMsg()
        checkState()
        showBusy()
    }

    function toggleOfflineMode() {
        coreModel.toggleOfflineMode()
    }

    function checkMessages() {
        var msgObj = msgQueue.shift() || null
        if (msgObj === null) {
            return
        }

        msgDialog.messageType = msgObj[0]
        msgDialog.msg = msgObj[1]
        msgDialog.btnText = msgObj[2] || qsTr('OK')
        msgDialog.visible = true
    }
    
    function showMessageDialog(msgType, msg, btnText) {
        msgQueue.push([msgType, msg, btnText])
        if (msgDialog.visible) {
            return
        }

        checkMessages()
    }



    FontLoader { source: "fonts/Lato-Regular.ttf"}
    FontLoader { source: "fonts/Audiowide-Regular.ttf"}

    HelloScreen {
        id: helloScreen
        visible: coreModel.show_hello
        anchors.fill: parent
        z: 100
    }
    
    MessageDialog {
        id: msgDialog
        anchors.fill: parent
        z: 110
        visible: false
        onOkClicked: {
            checkMessages()
        }
    }

    ConfirmDialog {
        id: confirmMsgDialog
        anchors.centerIn: parent
        z: 110
        visible: false
        property string callback: ""
        onYes: {
            if (callback != "") {
                var dynamicObject = Qt.createQmlObject('import QtQuick 2.7; QtObject { function callback() {' + callback + '}}', confirmMsgDialog, 'confirm_callback')
                dynamicObject.callback()
                dynamicObject.destroy()
                callback = ""
            }
        }

    }

    ConfirmDialog {
        id: networkConfirm
        width: AppStyle.rootItemWidth
        height: AppStyle.rootItemHeight
        anchors.centerIn: parent
        z: 105
        visible: rootItem.networkConfirmVisible
        hideOnBtnClick: false
        titleText: qsTr("Warning")
        titleColor: AppStyle.orange
        msg: qsTr("Can't connect to network")
        yesText: qsTr("Wi-Fi Setup")
        noText: qsTr("Offline mode")
        onYes: {
            if (settingsOpened) {
                return
            }

            rootItem.showSetting()
            rootItem.pushView("SelectWifiView.qml")
        }
        onNo: {
            if (coreModel.offlineMode)
                return

            coreModel.toggleOfflineMode()
            if (coreModel.state === Constants.States.REGISTRATION) {
                showBusy()
            }
            rootItem.checkState()
        }
    }

    ConfirmDialog {
        id: updateConfirm
        //        width: AppStyle.rootItemWidth
        //        height: AppStyle.rootItemHeight
        //        anchors.centerIn: parent
        z: 105
        visible: false
        titleText: qsTr("Info")
        property string version: ""
        msg: qsTr("Software update is available.\nAre you sure you wish to update?\n(This will cause a reboot)")
        yesText: qsTr("Update")
        noText: qsTr("Cancel")
        onYes: {
            pushView("UpdateScreen.qml")
            rootItem.isUpdating = true
            coreModel.performUpdate()
        }
    }

    Rectangle {
        id: rootRect
        anchors {
            top: parent.top
            bottom: keyboard.visible ? keyboard.top : parent.bottom
            left: parent.left
            right: parent.right
        }
        color: AppStyle.background

        AppHead {
            id: appHead
            visible: true
            height: appHead.visible ? AppStyle.headHeight : 0
            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }

            Behavior on height { PropertyAnimation { duration: 250 } }
        }

        Item {
            id: loaderItem
            z: 100
            anchors.fill: parent
            visible: false

            Rectangle {
                id: loaderLayer
                anchors.fill: parent
                color: "black"
                opacity: 0.75
            }

            AppBusyIndicator {
                id: loaderIndicator
                anchors.centerIn: parent
                anchors.verticalCenterOffset: -AppStyle.headHeight
                width: Math.min(parent.width, parent.height) * 0.25
                height: Math.min(parent.width, parent.height) * 0.25
                running: parent.visible
                z: 101
            }

            Text {
                id: loaderText
                anchors {
                    horizontalCenter: parent.horizontalCenter
                    top: loaderIndicator.bottom
                }
                width: loaderLayer.width * 0.5
               // text: "some very long tttext dasndkasndjasnjdasjldnasjldalsdljasnldjasnjdnasdn asjldnasld asjndl nlasdn lasld naldn lasndkl asnld nlasn dla"
                text: ""
                color: AppStyle.textColor
                font.pixelSize: AppStyle.headHeight * 0.5
                horizontalAlignment: Text.AlignHCenter
                wrapMode: Text.WordWrap
                z: 101
            }

            MouseArea {
                anchors.fill: parent
            }
        }

        Rectangle {
            id: busyLayer
            color: "black"
            opacity: 0.25
            visible: !settingsOpened || !loaderLayer.visible
            z: settingsOpened ? 0 : 50

            anchors {
                top: appHead.bottom
                bottom: parent.bottom
                left: parent.left
                right: parent.right
            }

            AppBusyIndicator {
                anchors.centerIn: parent
                visible: !settingsOpened
                width: Math.min(parent.width, parent.height) * 0.25
                height: Math.min(parent.width, parent.height) * 0.25
                running: parent.visible
            }

            MouseArea {
                anchors.fill: parent
            }
        }

        StackView {
            id: stack
            visible: true
            initialItem: "TokenView.qml"
            anchors {
                top: appHead.bottom
                bottom: parent.bottom
                left: parent.left
                right: parent.right
                topMargin: AppStyle.defaultMargin * 0.6
                bottomMargin: keyboard.visible ? 0 : AppStyle.defaultMargin * 0.6
                leftMargin: AppStyle.defaultMargin
                rightMargin: AppStyle.defaultMargin
            }
        }
    }

    signal keyPressed(string key)
    signal enterPressed()

    Keyboard {
        id: keyboard
        height: keyboard.visible ? parent.height * 0.6 : 1
        keyHeight: keyboard.visible ? height / 4 : 0
        visible: false
        z: 10
        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }

        onEnterClicked: {
            rootItem.enterPressed()
        }

        onHideClicked: {
            keyboard.visible = false
        }

        onKeyClicked: {
            rootItem.keyPressed(key)
        }

        Behavior on height { PropertyAnimation { duration: 250 } }
    }

    function checkState() {
        console.log("Current State:" + coreModel.state)

        // ignoring while setting menu is opened
        if (settingsOpened || isUpdating) {
            return
        }

        // just show busy layer
        if (coreModel.state === Constants.States.CONNECTING) {
            showBusy()
            return
        }

        hideBusy()
        hideKeyboard()
        //The idea: view has objectName eqaul to model state. So if we already have this view on stack just pop to it, if not push it
        var stackItem = stack.find(function(item) {
            return item.objectName === coreModel.state;
        })

        if (stackItem !== null) {
            stack.pop(stackItem)
            return
        }

        switch (coreModel.state) {
        case Constants.States.REGISTRATION:
            stack.push("TokenView.qml")
            break
        case Constants.States.IDLE:
            stack.push("PrinterReadyView.qml")
            break
        case Constants.States.LOCAL:
            stack.push("LocalPrintFileList.qml")
            break
        case Constants.States.PRINTING:
            if (stack.currentItem.objectName !== Constants.States.DOWNLOADING) {
                stack.push("PrintView.qml")
            }
            break
        case Constants.States.DOWNLOADING:
            if (stack.currentItem.objectName !== Constants.States.PRINTING) {
                stack.push("PrintView.qml")
            }
            break
        case Constants.States.PAUSED:
            if (stack.currentItem.objectName !== Constants.States.PRINTING) {
                stack.push("PrintView.qml")
            }
            break
        }
    }

    Connections {
        target: coreModel
        onStateChanged: {
            checkState()
        }
    }

    Connections {
        target: networkModel
        onOn_connectConnected: {
            rootItem.hideBusy()
            rootItem.hideSetting()
        }
        onOn_networkConnectChanged: {
            rootItem.hideBusy()
            //networkModel.scanWifi()
        }
        onOn_connectError: {
            rootItem.hideBusy()
        }
    }

    signal coreMsgCatched(string type, string msg, string btnText)
    signal coreConfirmCatched(string title, string msg, string yesBtnText, string noBtnText, string callback)
    signal coreUpdateChecked(string version)
    signal coreUpdateError(string error)
    signal coreShowProcessingCatched(string msg, real oppacity)
    signal hideProcessingCatched()
    signal hideSettingCatched()

    onCoreShowProcessingCatched: {
        loaderItem.visible = true
        loaderLayer.opacity = oppacity
        if (oppacity == 0.0) {
            loaderLayer.opacity = 0.75
        }
        loaderText.text = msg
    }

    onHideProcessingCatched: {
        loaderItem.visible = false
        loaderLayer.opacity = 0.75
        loaderText.text = ""
    }

    onHideSettingCatched: {
        hideSetting()
    }

    onCoreMsgCatched: {
        showMessageDialog(type, msg, btnText)
    }

    onCoreConfirmCatched: {
        if (confirmMsgDialog.visible) {
            return
        }
        confirmMsgDialog.titleText = title
        confirmMsgDialog.msg = msg
        confirmMsgDialog.yesText = yesBtnText != "" ? yesBtnText: qsTr("Yes")
        confirmMsgDialog.noText = noBtnText !== "" ? noBtnText: qsTr("Yes")
        confirmMsgDialog.callback = callback
        confirmMsgDialog.visible = true
    }

    onCoreUpdateChecked: {
        console.log("onCoreUpdateChecked: " + version)
        if (version === "") {
            showMessageDialog(Constants.MessageTypes.INFO, qsTr("Already up to date"), "OK")
            return
        }

        updateConfirm.version = version
        updateConfirm.visible = true
    }

    onCoreUpdateError: {
        showMessageDialog(Constants.MessageTypes.ERROR, error, "OK")
        rootItem.isUpdating = false
        rootItem.showHead()
        rootItem.checkState()
    }

    Component.onCompleted: {
        coreModel.showMessage.connect(coreMsgCatched)
        coreModel.showConfirm.connect(coreConfirmCatched)
        coreModel.on_updateCheck.connect(coreUpdateChecked)
        coreModel.on_updateError.connect(coreUpdateError)
        coreModel.showProcessing.connect(coreShowProcessingCatched)
        coreModel.hideProcessing.connect(hideProcessingCatched)
        coreModel.hideSettings.connect(hideSettingCatched)
        checkState()
        //networkModel.scanWifi()

        // coreConfirmCatched('Test', "Should I show message 'Test Done'?", "Sure", "Not sure", 'coreModel.showMessage(null, "Test Done", null)')

        //        showMessageDialog(Constants.MessageTypes.INFO, "Some very long information\n Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy text ever since the 1500s, when an unknown printer took a galley of type and scrambled it to make a type specimen book. It has survived not only five centuries, but also the leap into electronic typesetting, remaining essentially unchanged. It was popularised in the 1960s with the release of Letraset sheets containing Lorem Ipsum passages, and more recently with desktop publishing software like Aldus PageMaker including versions of Lorem Ipsum.", "OK")
    }
}
