// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.2
import Qt.labs.folderlistmodel 2.13

import AppStyle 1.0
import AppControls 1.0
import AppViews 1.0
import "constants.js" as Constants

ListPage {
    id: selectFileView
    objectName: Constants.States.LOCAL

    property url topFolder: "file://" + filesModel.rootFileDir

    function formatBytes(bytes, decimals) {
        if(bytes === 0) return '0 Bytes';
        var k = 1024,
        dm = decimals <= 0 ? 0 : decimals || 2,
        sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'],
        i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
    }

    FolderListModel {
        id: folderListModel
        rootFolder: topFolder
        folder: topFolder
        //nameFilters: ["*.gcode"]
        nameFilters: ["*.gcode", "*.sl1", "*.3dposdlp", "*.g"]
        showDirsFirst: true
    }

    pageBody: ListView {
        id: listView
        clip: true
        anchors.fill: parent
        currentIndex: -1
        property bool showHead: false
        property bool scrollBarEnabled: listView.contentHeight > listView.height
        property real defaultItemHeight: 64
        property int subFoldersCount: 0

        ScrollBar.vertical: ScrollBar {
            id: scroll
            policy: ScrollBar.AlwaysOn
            visible: listView.scrollBarEnabled
            parent: listView.parent
            anchors {
                top: parent.top
                left: parent.right
                bottom: parent.bottom
            }

            contentItem: Rectangle {
                implicitWidth: 7
                implicitHeight: 100
                color: scroll.pressed ? AppStyle.theme.pressedColor : AppStyle.theme.foreground
            }
        }

        model: folderListModel

        header: Rectangle {
            id: headDotDot
            width: listView.width
            height: visible ? listView.defaultItemHeight : 0
            color: headDotDotMouseArea.pressed ? AppStyle.theme.foreground : "transparent"
            visible: listView.subFoldersCount > 0
            border {
                color: AppStyle.theme.listBorderColor
                width: 1
            }

            Text {
                id: dotdotText
                anchors {
                    left: parent.left
                    leftMargin: AppStyle.theme.defaultMargin * 0.6
                }
                height: listView.defaultItemHeight
                font.pixelSize: listView.defaultItemHeight * 0.45
                text: ".."
                color: AppStyle.theme.textColor
                horizontalAlignment: Text.AlignLeft
                verticalAlignment: Text.AlignVCenter
                elide: Text.ElideRight
            }

            MouseArea {
                id: headDotDotMouseArea
                anchors.fill: parent
                onClicked: {
                    listView.subFoldersCount--
                    listView.currentIndex = -1
                    var folderStr = folderListModel.folder.toString()
                    folderListModel.folder = folderStr.split('/').slice( 0, -1 ).join('/');
                    if (folderListModel.folder.toString().split("//")[1] === folderListModel.rootFolder.toString()) {
                        console.log("return to parent dir")
                    }
                }
            }
        }

        delegate: Rectangle {
            id: fileListDelegate
            width: listView.width
            height: visible ? listView.defaultItemHeight : 0
            color: (fileListDelegateMouseArea.pressed || fileListDelegate.ListView.isCurrentItem) ? AppStyle.theme.foreground : "transparent"
            visible: !filesModel.isLink(folderListModel.folder.toString().slice(7) + "/" + fileName)
            border {
                color: AppStyle.theme.listBorderColor
                width: 1
            }

            Row {
                id: fileListRow
                height: parent.height
                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                    rightMargin: 30
                    leftMargin: 30
                    verticalCenter: parent.verticalCenter
                }

                Text {
                    id: filenameText
                    height: fileListRow.height
                    width: fileListRow.width * 0.75
                    font.pixelSize: fileListRow.height * 0.35
                    text: fileName
                    color: AppStyle.theme.textColor
                    horizontalAlignment: Text.AlignLeft
                    verticalAlignment: Text.AlignVCenter
                    elide: Text.ElideRight
                }

                Text {
                    id: sizeText
                    width: fileListRow.width * 0.25
                    height: fileListRow.height
                    font.pixelSize: fileListRow.height * 0.35
                    text: fileIsDir ? qsTr("Folder") : formatBytes(fileSize)
                    color: AppStyle.theme.textColor
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            MouseArea {
                id: fileListDelegateMouseArea
                anchors.fill: parent
                onClicked: {
                    if (fileIsDir) {
                        listView.subFoldersCount++
                        listView.currentIndex = -1
                        folderListModel.folder += "/" + fileName
                        return
                    }

                    listView.currentIndex = index
                }
            }

            Behavior on height { PropertyAnimation { duration: 250 } }
        }

    }

    footer.nextBtnText: qsTr("Print")
    footer.nextBtnVisible: listView.currentIndex !== -1

    footer.onBackClicked: {
        rootItem.clearHeadMsg()
        rootItem.back()
    }

    footer.onNextClicked: {
        var filePath = folderListModel.get(listView.currentIndex, "filePath")
        console.log("sending file to print: " + filePath)
        filesModel.printFile(filePath)
    }

    Component.onCompleted: {
        print("Current folder: " + folderListModel.parentFolder)
        rootItem.setHeadMsg(qsTr("SD card"))
    }
}
