// Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019.
// Built in San Francisco.

// This software is distributed under a commercial license for personal,
// educational, corporate or any other use.
// The software as a whole or any parts of it is prohibited for distribution or
// use without obtaining a license from 3D Control Systems, Inc.

// All software licenses are subject to the 3DPrinterOS terms of use
// (available at https://www.3dprinteros.com/terms-and-conditions/),
// and privacy policy (available at https://www.3dprinteros.com/privacy-policy/)

import QtQuick 2.7
import QtQuick.Controls 2.4
import AppStyle 1.0
import AppDialogs 1.0
import "../constants.js" as Constants

/*
    Filament wizzard root item that manages wizzard pages
*/

Item {
    id: filamentWizardRoot

    enum WizardType {
        Load,
        Unload,
        EmergencyUnloadAll,
        CleanPrintHead
    }

    property int wizardType: FilamentWizard.WizardType.Load
    readonly property alias stackViewDepth: filamentWizardStack.depth


    readonly property string lastMaterialFilePath: filesModel.settingsDir + "/currentMaterials"

    property var materialList: ["", "", "", ""]

    function back() {
        filamentWizardStack.pop()
    }

    function pushView(qml, props) {
        props = props || {};
        return filamentWizardStack.push(qml, props)

    }

    function cancelWizard() {
        cancelConfirm.visible = true
    }

    function startTestTimer() {
        testTimer.start()
    }

    function saveFile(fileUrl, text) {
        return filesModel.writeFile(fileUrl, text)
    }

    function openFile(fileUrl) {
        return filesModel.readFile(fileUrl)
    }

    function getCurrentMaterial() {
        return openFile(lastMaterialFilePath)
    }

    function getCurrentTemp() {
        return openFile(lastTempFilePath)
    }

    function saveLastMaterial(text) {
        return saveFile(lastMaterialFilePath, text)
    }

    function saveLastTemp(text) {
        return saveFile(lastTempFilePath, text)
    }

    function loadMaterials(json_str) {
      let materials = ["", "", "", ""];
      try {
        const data = JSON.parse(json_str);
        if (Array.isArray(data)) {
          materials = data.slice(0, 4);
          while (materials.length < 4) {
            materials.push("");
          }
        }
      } catch (e) {
        console.error("Invalid JSON string", e);
      }
      return materials;
    }

    function getCurrentMaterialList() {
        return loadMaterials(openFile(lastMaterialFilePath))
    }

    function saveMaterialList() {
        return saveLastMaterial(JSON.stringify(filamentWizardRoot.materialList))
    }

    function cleanMaterialsSlotStorage()
    {
        let materials = ["", "", "", ""];
        return saveLastMaterial(JSON.stringify(materials))
    }

    ConfirmDialog {
        id: cancelConfirm
        z: 105
        anchors {
            verticalCenterOffset: -AppStyle.theme.headHeight * 0.5
            centerIn: parent
        }
        visible: false
        msg: qsTr("Cancel filament load/unload wizard?")
        onYes: {
            //Cancel Wizard
//            printerModel.executeGcodeLine('M98 P"finish.g"')
            printerModel.executeGcodeLine('M108')
            rootItem.back()
        }
    }

    StackView {
        id: filamentWizardStack
        visible: true
        anchors.fill: parent
    }

    signal gcodeResponse(string line, int resultCode)

    Timer {
        id: testTimer
        interval: 1500; running: false; repeat: false
        onTriggered: { gcodeResponse("ok", true); gcodeResponse("ok", true) }
    }

    onGcodeResponse: function(line, resultCode) {
        if (resultCode === 0) {
            console.log("Success response: " + line)
            filamentWizardStack.currentItem.footer.visible = true
            return
        }

        if (resultCode > 0) {
            var errorMsg = qsTr("Gcode error response '%1'").arg(line)
            rootItem.showMessageDialog(Constants.MessageTypes.ERROR, errorMsg, "OK")
            printerModel.executeGcodeLine('M108')
            rootItem.hideSetting()
        }

    }

    Component.onCompleted: {
        printerModel.localModeOn()
        printerModel.enableResponseCallbacks()
        printerModel.onOn_responseCallback.connect(gcodeResponse)
        materialList = getCurrentMaterialList()
        if (wizardType === FilamentWizard.WizardType.EmergencyUnloadAll || wizardType === FilamentWizard.WizardType.CleanPrintHead) {
            filamentWizardStack.push("HeatingInfo.qml")
        } else if (wizardType === FilamentWizard.WizardType.Load) {
            filamentWizardRoot.pushView("SelectFilamentSlotPage.qml")
        } else if (wizardType === FilamentWizard.WizardType.Unload){
            filamentWizardStack.push("SelectFilamentSlotPage.qml")
        }
    }

    Component.onDestruction: {
        printerModel.localModeOff()
        printerModel.disableResponseCallbacks()
        printerModel.executeGcodeLine('M108')
    }
}
