#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import time
import signal
import platform

import log
import paths
import version
import config
paths.init_path_to_libs()
from plugin_controller import PluginController
from user_login import UserLogin
from rights import RightsCheckerWaiter
from updater import Updater
from camera_controller import CameraController
# from usb_detect import USBDetector
from u3_detect import U3Detector
# from network_detect import NetworkDetector
from client_detector import ClientDetector
from printer_interface import PrinterInterface


class App(object):

    LOG_TIMESTAMP_PERIOD = 30
    VIRTUAL_PRINTER_USB_INFO = {"VID" : "ZZZZ", "PID": "ZZZZ", "SNR": "0", "COM": None}
    BROWSER_OPENING_DELAY = 2

    @log.log_exception
    def __init__(self):
        config.Config.instance().set_app_pointer(self)
        self.logger = log.create_logger('', log.LOG_FILE)
        self.logger.info("Starting 3DPrinterOS client. Version %s. Build %s." % (version.version, version.build))
        self.logger.info('Operating system: ' + platform.system() + ' ' + platform.release())
        self.logger.info('Server: ' + config.get_settings()['URL'])
        self.printer_interfaces = []
        self.virtual_printer_enabled = False
        self.stop_flag = False
        self.restart_flag = False
        self.closing_status = []
        self.plugin_controller = PluginController(self)
        self.rights_checker_waiter = RightsCheckerWaiter(self)
        self.user_login = UserLogin(self)
        signal.signal(signal.SIGINT, self.intercept_signal)
        signal.signal(signal.SIGTERM, self.intercept_signal)
        self.init_interface()
        self.wait_and_open_browser()
        detector = U3Detector()
        self.detected_printer = None
        self.detects_left = 20
        while not self.stop_flag:
            self.detected_printer = detector.detect_printer()
            if self.detected_printer:
                self.logger.info("Printer was detected: " + str(self.detected_printer))
                break
            self.detects_left -= 1
            if self.detects_left == 0:
                self.quit()
                return
            self.logger.info("Printer detects count left: " + str(self.detects_left))
            time.sleep(5)
        self.client_detector = ClientDetector(self)
        if self.user_login.wait():
            if self.rights_checker_waiter.wait():
                self.updater = Updater(self)
                config.Config.instance().set_profiles(self.user_login.profiles)
                self.virtual_printer_enabled = config.get_settings()['virtual_printer']['enabled']
                self.camera_controller = CameraController(self.user_login.user_token, self.user_login.mac)
                # self.usb_detector = USBDetector()
                # self.network_printers_detector = NetworkDetector(self)
                self.main_loop()
        self.quit()

    @staticmethod
    def open_browser():
        if config.get_settings()['web_interface']['browser_opening_on_start']:
            import webbrowser
            import logging
            address = "http://%s:%d" % (config.LOCAL_IP, config.get_settings()['web_interface']['port'])
            logger = logging.getLogger()
            logger.info("Opening browser tab %s" % address)
            webbrowser.open(address, 2, True)

    def wait_and_open_browser(self):
        start = time.time()
        self.logger.info("Waiting for timeout or login to open browser...")
        while time.time() < start + self.BROWSER_OPENING_DELAY and not self.user_login.user_token:
            time.sleep(0.01)
        self.logger.info("...done")
        App.open_browser()

    def init_interface(self):
        if config.get_settings()['web_interface']['enabled']:
            from web_interface import WebInterface
            self.web_interface = WebInterface(self)
            self.web_interface.start()
            self.logger.debug("Waiting for webserver to start...")
            while not self.web_interface.server:
                time.sleep(0.01)
                if self.stop_flag:
                    return
            self.logger.debug("...server is up and running.")

    def intercept_signal(self, signal_code, frame):
        self.logger.warning("SIGINT or SIGTERM received. Closing 3DPrinterOS Client version %s_%s" % \
                (version.version, version.build))
        self.stop_flag = True

    def get_all_network_profiles(self):
        self.logger.info("Getting all network printers profiles")
        while not hasattr(self, "user_login"):
            time.sleep(0.1)
        while not getattr(self.user_login, "profiles"):
            time.sleep(0.1)
        return [x for x in self.user_login.profiles if x.get('network_detect')]

    def toggle_virtual_printer(self):
        self.virtual_printer_enabled = not self.virtual_printer_enabled
        settings = config.get_settings()
        settings['virtual_printer']['enabled'] = self.virtual_printer_enabled
        config.Config.instance().save_settings(settings)

    @log.log_exception
    def main_loop(self):
        last_time = 0
        while not self.stop_flag:
            self.updater.check_update_timer()
            # detected_printers = self.u3_detector.get_printers_list()
            # detected_printers = [{'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '00U3'}]
            detected_printers = []
            if self.detected_printer:
                detected_printers.append(self.detected_printer)
            if detected_printers is not None:
                if self.virtual_printer_enabled:
                    detected_printers.append(self.VIRTUAL_PRINTER_USB_INFO)
                # detected_printers.extend(self.network_printers_detector.get_printers_list())
                self.connect_new_printers(detected_printers)
                for pi in self.printer_interfaces:
                    if pi.usb_info not in detected_printers:
                        if not pi.forced_state == "error":
                            pi.register_error(99, "Printer is no longer detected as device", is_blocking=True)
                    if not pi.is_alive():
                        self.logger.info('Removing %s from printer list' % pi.usb_info)
                        self.printer_interfaces.remove(pi)
            if not self.stop_flag:
                time.sleep(config.get_settings()['main_loop_period'])
                now = time.time()
                if last_time + self.LOG_TIMESTAMP_PERIOD < now:
                    last_time = now
                    self.logger.debug(time.strftime("%d %b %Y %H:%M:%S", time.localtime()))

    def connect_new_printers(self, detected_printers):
        currently_connected_usb_info = [pi.usb_info for pi in self.printer_interfaces]
        for usb_info in detected_printers:
            if usb_info not in currently_connected_usb_info:
                pi = PrinterInterface(self, usb_info)
                currently_connected_usb_info.append(pi.usb_info)
                pi.start()
                self.printer_interfaces.append(pi)

    def shutdown_web_interface(self):
        self.logger.debug("Waiting web interface server to shutdown...")
        try:
            self.web_interface.server.shutdown()
            self.web_interface.join()
        except AttributeError:
            pass
        self.logger.debug("...done")
        self.logger.debug("Waiting websocket interface server to shutdown...")
        self.logger.debug("...done")

    def close_module(self, state, closing_function):
        self.logger.info(state)
        self.closing_status.append(state)
        closing_function()
        self.closing_status[-1] += 'done'
        time.sleep(0.5)

    def register_error(self, code, message, is_blocking=False, is_info=False):
        #error = { "code": code, "message": message, "is_blocking": is_blocking }
        self.logger.warning("Error N%d. %s" % (code, message))

    def quit(self): # TODO reduce copy-paste
        self.logger.info("Starting exit sequence...")
        if hasattr(self, 'client_detector'):
            self.close_module('Closing client detector...', self.client_detector.close)
        if hasattr(self, 'camera_controller'):
            self.close_module('Closing camera...', self.camera_controller.stop_camera_process)
        for pi in self.printer_interfaces:
            pi.close()
        for pi in self.printer_interfaces:
            printer_name = 'undefined printer'
            if pi.printer_profile and pi.printer_profile.get('name'):
                printer_name = pi.printer_profile['name']
            state = 'Closing ' + printer_name + '...'
            self.close_module(state, pi.join)
        self.close_module('Closing Plugins...', self.plugin_controller.close)
        time.sleep(0.2) # need to reduce logging spam
        self.shutdown_web_interface()
        time.sleep(0.2)  # to close web server threads
        self.plugin_controller.close()
        self.logger.info("...everything is correctly closed.")
        self.logger.info("Goodbye ;-)")
        for handler in self.logger.handlers:
            handler.flush()
