#!/usr/bin/env python
#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import re
import sys
import tempfile
import subprocess
import zipfile

import paths
import platforms


windows_process_name = 'pythonw.exe'
unix_process_name_re = re.compile(".*python.*(launcher\.py|app\.py)")
pid_file_name = "3dprinteros.pid"
tmp_path = tempfile.gettempdir()
pid_file_path = os.path.join(tmp_path, pid_file_name)
update_file_path = paths.UPDATE_FILE_PATH
update_script_file_name = 'update_script.py'


def install_update():
    print('Updating 3DPrinterOS')
    try:
        z = zipfile.ZipFile(update_file_path)
        z.extractall()
        z.close()
    except zipfile.BadZipfile:
        os.remove(update_file_path)
        print('Update failed!\nReason: bad zip file')
        return True
    except Exception as e:
        print('Update failed!\nReason: error while extracting.\nDetails: ' + str(e))
    else:
        os.remove(update_file_path)
        print('Update successful!')
        try:
            from importlib import reload
            reload(paths)
        except Exception as e:
            print('Unable to reload modules. Error:' + str(e))
        open(os.path.join(paths.CURRENT_SETTINGS_FOLDER, 'update_complete'), "w")
        if os.path.exists(update_script_file_name):
            subprocess.call([sys.executable, update_script_file_name])
            os.remove(update_script_file_name)
        return True


def run_command(command):
    process = subprocess.Popen(command, shell=True, stdout=subprocess.PIPE)
    output = process.stdout.readlines()
    process.stdout.close()
    return output


def get_nix_processes():
    return run_command('ps ax')


def get_win_processes():
    processes_list = run_command("tasklist /svc")
    return [proc for proc in processes_list if len(proc.split()) > 1]


def check_process_name_by_pid(pid):
    print("Checking process name")
    if sys.platform.startswith('win'):
        for task in get_win_processes():
            words = task.split()
            if len(words) > 1:
                current_task_pid = words[1]
                if current_task_pid == pid:
                    return words[0] == windows_process_name
    elif sys.platform.startswith('linux') or sys.platform.startswith('darwin'):
        for task in get_nix_processes():
            words = task.split()
            if words[0] == pid and len(words) > 5:
                process = words[4] + words[5]
                return unix_process_name_re.match(process.decode('utf-8'))
    else:
        raise RuntimeError("Your OS is not supported by 3DPrinterOS")


def get_process_list():
    print("Getting process list")
    if sys.platform.startswith('win'):
        processes = get_win_processes()
        pids = [x.split()[1] for x in processes]
    elif sys.platform.startswith('linux') or sys.platform.startswith('darwin'):
        processes = get_nix_processes()
        pids = [x.split()[0] for x in processes]
    else: raise RuntimeError("Your OS is not supported by 3DPrinterOS")
    return pids


def execute_custom_args():
    if len(sys.argv) > 1:
        args = sys.argv[1:]
        if '--restore_default_settings' in args:
            import config
            os.remove(config.Config.USER_SETTINGS_FILE)
            print("Default settings were restored")
        if '--logout' in args:
            import user_login
            user_login.logout()
            print("Logout successful")
        if '--clear_logs' in args:
            import log
            log.clear_logs()
            print("Log files were erased")
        if '--erase_all_data':
            for root, dirs, files in os.walk(paths.CURRENT_SETTINGS_FOLDER, topdown=False):
                for name in files:
                    os.remove(os.path.join(root, name))
                for name in dirs:
                    os.rmdir(os.path.join(root, name))
            os.rmdir(paths.CURRENT_SETTINGS_FOLDER)
            print("3DPrinterOS's data folder was erased")
        sys.exit(0)


def restart():
    print("Restarting 3DPrinterOS")
    python = sys.executable
    args = [python, python, __file__]
    if platforms.get_platform() == "win":
        args[1] = '"' + args[1] + '"'
        args[2] = '"' + args[2] + '"'
    os.execl(*args)


def check_system_timestamp():
    print("Checking system timestamp")
    shell = '/bin/bash'
    args = [shell, os.path.join(os.path.dirname(os.path.abspath(__file__)), 'timeadjust.sh')]
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    stdout, _ = p.communicate()
    stdout = stdout.decode('utf-8')
    print("Result: code=%d\n%s" % (p.returncode, stdout))


def run():
    execute_custom_args()
    check_system_timestamp()
    print("Launching 3DPrinterOS")
    with open(pid_file_path, "w") as f:
        f.write(str(os.getpid()))
    if os.path.exists(update_file_path):
        if not install_update():
            "Update failed. Can't start."
            return
    import app
    printeros_app = app.App()
    os.remove(pid_file_path)
    if printeros_app.restart_flag:
        restart()


def open_browser_tab():
    import app
    app.App.open_browser()


if __name__ == "__main__":
    try:
        f = open(pid_file_path)
    except IOError:
        run()
    else:
        pid_in_file = bytes(f.read(), encoding='utf-8')
        f.close()
        if pid_in_file in get_process_list() and check_process_name_by_pid(pid_in_file):
            print("3DPrinterOS is already running - opening browser window")
            open_browser_tab()
        else:
            run()




