#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import ssl
import logging
import threading
import http.server
from socketserver import ThreadingMixIn

import config
from . import page_former
from . import web_interface_handler
import log



class ThreadedHTTPServer(ThreadingMixIn, http.server.HTTPServer):
    """ This class allows to handle requests in separated threads.
        No further content needed, don't touch this. """


class WebInterface(threading.Thread):

    @staticmethod
    def get_web_content_files(path=None):
        binary_files_extensions = ('jpg', 'png')
        if not path:
            path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'web_content_files')
        web_content_files = {}
        for name in os.listdir(path):
            file_path = os.path.join(path, name)
            if os.path.isdir(file_path):
                web_content_files[name] = WebInterface.get_web_content_files(file_path)
            else:
                read_mode = 'r'
                if name.split(".")[-1] in binary_files_extensions:
                    read_mode += 'b'
                try:
                    with open(file_path, read_mode) as f:
                        web_content_files[name] = f.read()
                except Exception as e:
                    logger = logging.getLogger(__name__)
                    error = 'Error while reading file "' + name + '": ' + str(e)
                    logger.error(error)
                    raise RuntimeError(error)
        return web_content_files

    def __init__(self, app):
        self.logger = logging.getLogger(__name__)
        self.app = app
        self.server = None
        threading.Thread.__init__(self, name="WebInterface")

    def find_printer_interface(self, usb_info_string):
        for printer_interface in self.server.app.printer_interfaces:
            if str(printer_interface.usb_info) == usb_info_string:
                return printer_interface

    @log.log_exception
    def run(self):
        self.logger.info("Starting web server...")
        if config.get_settings()['remote_control']['web_server']:
            IP = config.REMOTE_IP
        else:
            IP = config.LOCAL_IP
        port = config.get_settings()['web_interface']['port']
        try:
            self.server = ThreadedHTTPServer((IP, port), web_interface_handler.WebInterfaceHandler)
            self.server.daemon_threads = True
            #self.server.socket = ssl.wrap_socket(self.server.socket, keyfile='server.key', certfile='server.crt', ca_certs=False, server_side=True)
        except Exception as e:
            self.logger.error(e)
        else:
            self.logger.info("...web server started")
            self.server.app = self.app
            self.server.web_content_files = self.get_web_content_files()
            self.server.page_former = page_former.PageFormer(self.server)
            self.server.find_printer_interface = self.find_printer_interface
            self.server.serve_forever()
            self.server.server_close()
            self.logger.info("Web server stop.")
