#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2017 3D Control Systems LTD

# Author: Maxim Kozlov <m.kozlov@3dprinteos.com>

import logging

import config
import u3api
from u3_sender import Sender


class U3Detector:
    AVAILABLE_PRINTERS = {
        'ultimaker 3': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '00U3'},
        'ultimaker 3 extended': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0U3E'},
        'ultimaker s5': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0US5'},
        'ultimaker s3': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0US3'},
        'ultimaker s7': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0US7'},
        'ultimaker s6': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0US6'},
        'ultimaker s8': {'COM': None, 'VID': 'ZZZZ', 'SNR': '0', 'PID': '0US8'},
    }

    def __init__(self):
        self.logger = logging.getLogger(__name__)

    def detect_printer(self, force_detect=False, save_to_config=True):
        if force_detect:
            printer_type = None
        else:
            printer_type = config.get_settings().get('printer_type')
            if isinstance(printer_type, str):
                printer_type = printer_type.lower()
            self.logger.info('Got printer type from config: ' + str(printer_type))
        if not printer_type or printer_type not in self.AVAILABLE_PRINTERS:
            self.logger.info('Detecting printer type using request to printer')
            con = u3api.Ultimaker3(Sender.PRINTER_IP, Sender.APPLICATION_NAME)
            url = Sender.BASE_API_URL + 'system/variant'
            self.logger.info('REQUEST: ' + url)
            try:
                result = con.get(url)
            except Exception as e:
                self.logger.info('Exception: ' + str(e))
                return None
            result.close()
            self.logger.info('RESPONSE: %d, %s' % (result.status_code, result.content))
            success = (result.status_code >= 200) and (result.status_code < 300)
            if success:
                printer_type = result.json()
                if isinstance(printer_type, str):
                    printer_type = printer_type.lower()
            if not success or printer_type not in self.AVAILABLE_PRINTERS:
                self.logger.error('Problem with printer type detection. Please use manual type selection')
                return None
        if save_to_config:
            settings = config.get_settings()
            settings['printer_type'] = printer_type
            config.Config.instance().save_settings(settings)
        return self.AVAILABLE_PRINTERS[printer_type]

if __name__ == '__main__':
    import sys
    import getopt
    opts, _ = getopt.getopt(sys.argv[1:], "fsh")
    force = False
    save = False
    for opt, arg in opts:
        if opt == '-h':
            print('u3_detect.py [-f] [-s]')
            sys.exit()
        elif opt == '-f':
            force = True
        elif opt == '-s':
            save = True
    detector = U3Detector()
    import log
    log.create_logger('')
    printer = detector.detect_printer(force, save)
    print("\nDetected printer:")
    print(printer)
