#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2017 3D Control Systems LTD

# Author: Maxim Kozlov <m.kozlov@3dprinteos.com>

import os
import time
import logging
import threading

import log
import u3api
import paths
import config
from base_sender import BaseSender


class Sender(BaseSender):

    PRINTER_IP = "127.0.0.1:80"
    BASE_API_URL = "api/v1/"
    APPLICATION_NAME = "3DPrinterOS Client"
    PRINTER_AUTH_FILE = os.path.join(paths.current_settings_folder(), 'auth_client.data')

    FILE_NAME = "3dprinteros.gcode"

    STATE_BOOTING = "booting"
    STATE_READY = "idle"
    STATE_PRINTING = "printing"
    STATE_ERROR = "error"
    STATE_MAINTENANCE = "maintenance"

    def __init__(self, parent, usb_info, profile):
        BaseSender.__init__(self, parent, usb_info, profile)
        self.verbose = config.get_settings()['verbose']
        self.pause_flag = False
        self.monitoring_stop = False
        self.operational_flag = False
        self.printing_flag = False
        self.bed_clear_flag = True
        self.lines_sent = 0
        self.total_gcodes = 0
        self.percent = 0
        self.upload_percent = 0
        self.can_print_gzip = True
        self.connection = u3api.Ultimaker3(self.PRINTER_IP, self.APPLICATION_NAME)
        self.send_and_wait_lock = threading.Lock()
        self.connect()
        try:
            self.connection.loadAuth(self.PRINTER_AUTH_FILE)
        except Exception as ex:
            self.logger.warn('Exception on loadAuth: ' + str(ex))
        self.monitoring_thread = None
        self.monitoring_thread_start()

    def connect(self):
        retries = 0
        while retries < 10 and not self.parent.stop_flag:
            result = self.request('system/name')
            if result[0]:
                self.logger.info("U3 printer's name: %s" % result[1])
                return True
            retries += 1
            self.logger.warning("U3 connect problem. Retry %d of 10" % (retries))
            time.sleep(5)
        raise RuntimeError("Cannot connect to printer's API and receive 200 response")

    def request(self, url, method="get", **kwargs):
        with self.send_and_wait_lock:
            success = False
            try:
                if self.verbose:
                    self.logger.info('REQUEST: '+url)
                result = self.connection.request(method, self.BASE_API_URL + url, **kwargs)
                if self.verbose:
                    self.logger.info('RESPONSE: %d, %s' % (result.status_code, result.content))
                success = (result.status_code >= 200) and (result.status_code < 300)
                return success, result.json() if len(result.content) > 0 else ''
            except Exception as e:
                self.logger.info('Exception: ' + str(e))
                return success, ''

    @log.log_exception
    def monitoring(self):
        self.logger.info("Monitoring thread started")
        count = 0
        last_status = None
        while not self.monitoring_stop and not self.stop_flag:
            printer_result = self.request('printer')
            if not printer_result[0]:
                if count < 5:
                    count += 1
                    self.logger.info("Printer not answering on monitoring requests. Retry number %d out of 5" % count)
                    continue
                self.operational_flag = False
                message = "Printer not answering in monitoring. Printer lost."
                self.parent.register_error(606, message, is_blocking=True)
                break
            count = 0
            self.parse_temperature(printer_result[1])
            status = printer_result[1]['status']
            self.operational_flag = True
            if status == self.STATE_PRINTING:
                print_job_result = self.request('print_job')
                if print_job_result[0] and print_job_result[1]:
                    print_job = print_job_result[1]
                else:
                    print_job = None
                if not self.printing_flag:
                    self.percent = 0
                    self.printing_flag = True
                if print_job:
                    job_state = print_job.get('state')
                    job_progress = int(print_job.get('progress')*100)
                    self.logger.info('job state: %s, progress: %f' % (job_state, job_progress))
                    if job_progress > 0:
                        self.percent = job_progress
                    self.pause_flag = (job_state == 'paused')
                    # When job_state is wait_user_action for finished and cleaned job we will have
                    # printing locally in Cloud. It is wrong, as printer is waiting for user's action.
                    # We will set bed_not_clear to show that some interaction is needed
                    if job_state == 'wait_cleanup' or\
                            (job_state == 'wait_user_action' and print_job.get('datetime_cleaned')):
                        self.bed_clear_flag = False
                    elif not self.bed_clear_flag and (job_state == 'printing' or self.pause_flag):
                        self.bed_clear_flag = True
            elif status == self.STATE_READY:
                if self.printing_flag:
                    self.pause_flag = False
                    self.printing_flag = False
                    self.bed_clear_flag = True
                    if self.percent < 100:
                        self.parent.register_error(607, "Cancelled manually", is_blocking=True)
            else:
                self.operational_flag = False
                if last_status != status:
                    if status == self.STATE_BOOTING:
                        self.parent.register_error(700, 'Printer booting', is_blocking=False)
                    elif status == self.STATE_MAINTENANCE:
                        self.parent.register_error(701, 'Printer maintenance', is_blocking=False)
                    else:
                        self.parent.register_error(702, 'Printer error', is_blocking=False)
            last_status = status
            time.sleep(1)
        self.logger.info("Monitoring thread stopped")

    def load_gcode_file(self, gcode_file):
        if self.operational_flag and not self.is_printing():
            return self.upload_gcodes_and_print(gcode_file)
        else:
            self.parent.register_error(604, "Printer already printing.", is_blocking=False)
            return False

    def upload_gcodes_and_print(self, gcode_file):
        self.monitoring_thread_stop()
        self.percent = 0
        self.upload_percent = 0
        self.lines_sent = 0

        # Send verify request prior print_job for more effective HTTP Digest authentication
        # Without it print_job request (full gcode file) is send twice
        self.request('auth/verify')

        filename = self.filename if self.filename else self.FILE_NAME
        if gcode_file.name.endswith('.gz'):
            filename = filename + '.gz'
        self.logger.info('Filename: %s' % filename)
        response = self.request('print_job', method="post", files={"file": (filename, gcode_file)})
        if not response[0]:
            message = "Error on print_job request: %s" % str(response[1])
            self.parent.register_error(703, message, is_blocking=True)
            return False

        self.logger.info("New Print job was successfully started")
        self.printing_flag = True
        time.sleep(2)
        self.monitoring_thread_start()
        return True

    def pause(self):
        if not self.pause_flag:
            return self.request('print_job/state', method='put', data={"target": "pause"})[0]

    def unpause(self):
        if self.pause_flag:
            return self.request('print_job/state', method='put', data={"target": "print"})[0]

    def cancel(self):
        if self.request('print_job/state', method='put', data={"target": "abort"})[0]:
            self.logger.info('Cancelled!')
        else:
            self.logger.info('Cancel error on M26')

    def is_operational(self):
        return self.operational_flag

    def is_paused(self):
        return self.pause_flag

    def is_printing(self):
        return self.printing_flag

    def is_bed_clear(self):
        return self.bed_clear_flag

    def get_percent(self):
        return self.percent

    def get_current_line_number(self):
        if self.printing_flag:
            return self.lines_sent
        else:
            return 0

    def parse_temperature(self, printer_info):
        tool_0_temp = float(printer_info['heads'][0]['extruders'][0]['hotend']['temperature']['current'])
        tool_0_target_temp = float(printer_info['heads'][0]['extruders'][0]['hotend']['temperature']['target'])
        tool_1_temp = float(printer_info['heads'][0]['extruders'][1]['hotend']['temperature']['current'])
        tool_1_target_temp = float(printer_info['heads'][0]['extruders'][1]['hotend']['temperature']['target'])
        platform_temp = float(printer_info['bed']['temperature']['current'])
        platform_target_temp = float(printer_info['bed']['temperature']['target'])
        self.temps = [round(platform_temp, 2), round(tool_0_temp, 2), round(tool_1_temp, 2)]
        self.target_temps = [round(platform_target_temp, 2), round(tool_0_target_temp, 2), round(tool_1_target_temp, 2)]
        return True

    def close(self):
        self.monitoring_thread_stop()
        self.stop_flag = True

    def monitoring_thread_start(self):
        self.monitoring_stop = False
        self.monitoring_thread = threading.Thread(target=self.monitoring)
        self.monitoring_thread.start()

    def monitoring_thread_stop(self):
        self.monitoring_stop = True
        if self.monitoring_thread:
            self.monitoring_thread.join()

if __name__ == "__main__":
    logging.basicConfig(level=logging.DEBUG)
    sender = None
    sender = Sender({}, {"VID": "ZZZZ", "PID": "00U3"}, {})
    try:
        time.sleep(3)
    except Exception:
        if sender:
            sender.close()
    try:
        time.sleep(10)
    except:
        pass
    if sender:
        sender.close()
