#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import sys
import time
import errno
import zipfile
import logging
import logging.handlers
import traceback
import platform
import os.path
import json

import paths
import requests
import http_client
import config
import version
import subprocess

SETTINGS_FOLDER = paths.current_settings_folder()
LOG_FILE = os.path.join(SETTINGS_FOLDER, "3dprinteros_client.log")
CAMERA_LOG_FILE = os.path.join(SETTINGS_FOLDER, "3dprinteros_camera.log")
EXCEPTIONS_LOG_FILE = os.path.join(SETTINGS_FOLDER, 'critical_errors.log')
REPORT_FILE_NAME = 'problem_report.txt'
REPORT_FILE_PATH = os.path.join(SETTINGS_FOLDER, REPORT_FILE_NAME)

LOG_FILE_SIZE = 1024*1024*5 #(5MB)
LOG_BACKUPS = 1 #total max long 0 + 1 = 50MB + 50MB = 100MB

def create_logger(logger_name, log_file_name=None):
    logger = logging.getLogger(logger_name)
    logger.setLevel(logging.DEBUG)
    std_handler = logging.StreamHandler(stream=sys.stdout)
    std_handler.setLevel(logging.DEBUG)
    logger.addHandler(std_handler)
    if log_file_name and config.get_settings().get('logging') and config.get_settings()['logging']['enabled']:
        if config.get_settings()['logging']['erase_on_start']:
            clear_logs()
        try:
            file_handler = logging.handlers.RotatingFileHandler(log_file_name, 'a', LOG_FILE_SIZE, LOG_BACKUPS)
            file_handler.setFormatter(logging.Formatter('%(levelname)s\t%(asctime)s\t%(threadName)s/%(funcName)s\t%(message)s'))
            file_handler.setLevel(logging.DEBUG)
            logger.addHandler(file_handler)
            print("File logger created: " + log_file_name)
        except Exception as e:
            print('Could not create log file because' + str(e) + '\n.No log mode.')
    return logger

def log_exception(func_or_method):
    def decorator(*args, **kwargs):
        try:
            result = func_or_method(*args, **kwargs)
        except SystemExit:
            pass
        except IOError as e:
            if e.errno != errno.EINTR:
                report_exception()
        except:
            report_exception()
        else:
            return result
    return decorator

def report_exception():
    trace = traceback.format_exc()
    try:
        logging.getLogger(__name__).error(trace)
    except:
        print(trace)
    with open(EXCEPTIONS_LOG_FILE, "a") as f:
        f.write(form_log_title() + time.ctime() + "\n" + trace + "\n")
    if config.get_settings()['logging']['auto_report_exceptions']:
        try:
            send_logs()
        except:
            logging.getLogger(__name__).error("Error while trying to send logs")
    sys.exit(1)

def compress_logs():
    logger = logging.getLogger(__name__)
    try:
        log_file_names = os.listdir(SETTINGS_FOLDER)
    except:
        logger.warning('No logs to pack')
    else:
        for name in log_file_names[:]:
            if not (name.startswith(os.path.basename(LOG_FILE))
                    or name == os.path.basename(EXCEPTIONS_LOG_FILE)
                    or name == REPORT_FILE_NAME):
                log_file_names.remove(name)
        if log_file_names:
            zip_file_name = time.strftime("%Y_%m_%d___%H_%M_%S", time.localtime()) + ".zip"
            zip_file_name_path = os.path.abspath(os.path.join(SETTINGS_FOLDER, zip_file_name))
            logger.info('Creating zip file : ' + zip_file_name)
            try:
                zf = zipfile.ZipFile(zip_file_name_path, mode='w')
                for name in log_file_names:
                    zf.write(os.path.join(SETTINGS_FOLDER, name), name, compress_type=zipfile.ZIP_DEFLATED)
                zf.close()
            except Exception as e:
                logger.warning("Error while creating logs archive " + zip_file_name + ': ' + str(e))
            else:
                return zip_file_name_path

def upload_compressed_logs(zip_file_path):
    logger = logging.getLogger(__name__)
    url = 'https://' + http_client.HTTPClient.URL + http_client.HTTPClient.token_send_logs_path
    try:
        data = {'user_token': config.get_app().user_login.user_token}
        logger.info('Sending logs to %s' % url)
        with open(zip_file_path, 'rb') as zip_file:
            files = {'file_data': zip_file}
            response = None
            if os.path.exists(REPORT_FILE_PATH):
                with open(REPORT_FILE_PATH, 'rb') as report_file:
                    files['report_file'] = report_file
                    response = requests.post(url, data=data, files=files)
                os.remove(REPORT_FILE_PATH)
    except Exception as e:
        return 'Error while sending logs: ' + str(e)
    else:
        if response:
            result = response.text
            logger.debug("Log sending response: " + result)
            if response.status_code != 200:
                return 'Error while uploading logs: response code is not 200 (OK)'
            try:
                answer = json.loads(result)
            except Exception as e:
                return 'Error while uploading logs: ' + str(e)
            if not answer.get('success'):
                return result

def send_logs():
    logger = logging.getLogger(__name__)
    if config.get_settings()['logging']['logs_sending']:
        zip_file_path = compress_logs()
        if not zip_file_path:
            return 'Error while packing logs'
        error = upload_compressed_logs(zip_file_path)
        os.remove(zip_file_path)
        if error:
            logger.warning(error)
            return error
        logger.debug('Logs successfully sent')
        return clear_logs()
    else:
        logger.warning("Can't send logs - disabled in config")

def clear_logs():
    logger = logging.getLogger(__name__)
    remove_old_logs()
    try:
        with open(LOG_FILE, 'w') as f:
            f.write(form_log_title())
        if os.path.exists(EXCEPTIONS_LOG_FILE):
            os.remove(EXCEPTIONS_LOG_FILE)
    except Exception as e:
        error = 'Error while clearing critical errors log: ' + str(e)
        logger.warning(error)
        return error
    logger.info('Logs successfully cleared')

def remove_old_logs():
    logger = logging.getLogger(__name__)
    for handler in logger.handlers:
        handler.do_rollover()
    for number in range(1, 10):
        ending = "." + str(number)
        try:
            os.remove(LOG_FILE + ending)
        except (OSError, IOError):
            break

def get_file_tail(file_path):
    if os.path.isfile(file_path):
        with open(file_path) as file:
            f = file.readlines()
        file_tail = []
        for line in range(-1,-100, -1):
            try:
                file_tail.append(f[line])
            except IndexError:
                break
        return file_tail

def form_log_title():
    version_string = "%s.%s_%s" % (version.version, version.build, version.branch)
    return 'Version %s\nOperating system:%s %s\n' % (version_string, platform.system(), platform.release())

def create_report(report_text):
    report_text = form_log_title() + '\n\n' + report_text
    paths.check_and_create_dirs(REPORT_FILE_PATH)
    with open(REPORT_FILE_PATH, 'wb') as f:
        f.write(bytes(report_text, 'utf-8'))

def report_problem(report_text):
    create_report(report_text)
    result = send_logs()
    return result

def open_settings_folder():
    path = os.path.abspath(SETTINGS_FOLDER)
    if sys.platform.startswith('darwin'):
        subprocess.Popen(['open', path], close_fds=True)
    elif sys.platform.startswith('linux'):
        subprocess.Popen(['xdg-open', path], close_fds=True)
    elif sys.platform.startswith('win'):
        subprocess.Popen(['explorer', path], close_fds=True)

# def tail(f, lines=200):
#     total_lines_wanted = lines
#     BLOCK_SIZE = 1024
#     f.seek(0, 2)
#     block_end_byte = f.tell()
#     lines_to_go = total_lines_wanted
#     block_number = -1
#     blocks = [] # blocks of size BLOCK_SIZE, in reverse order starting
#                 # from the end of the file
#     while lines_to_go > 0 and block_end_byte > 0:
#         if (block_end_byte - BLOCK_SIZE > 0):
#             # read the last block we haven't yet read
#             f.seek(block_number*BLOCK_SIZE, 2)
#             blocks.append(f.read(BLOCK_SIZE))
#         else:
#             # file too small, start from begining
#             f.seek(0,0)
#             # only read what was not read
#             blocks.append(f.read(block_end_byte))
#         lines_found = blocks[-1].count('\n')
#         lines_to_go -= lines_found
#         block_end_byte -= BLOCK_SIZE
#         block_number -= 1
#     all_read_text = ''.join(reversed(blocks))
#     return '\n'.join(all_read_text.splitlines()[-total_lines_wanted:])
