#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import sys
import time
import json
import logging
import threading

import log
import downloader
import http_client


class PrinterInterface(threading.Thread):

    DEFAULT_OPERATIONAL_TIMEOUT = 10
    COMMAND_REQUEST_PERIOD = 1.5

    def __init__(self, parent, usb_info):
        self.logger = logging.getLogger(__name__)
        self.creation_time = time.time()
        self.stop_flag = False
        self.stop_in_next_loop_flag = False
        self.usb_info = usb_info
        self.parent = parent
        self.printer = None
        self.printer_profile = {}
        self.downloader = None
        self.forced_state = "connecting"
        self.printer_name = ""
        self.groups = []
        self.last_error = {}
        self.errors = []
        self.errors_to_display = []
        self.local_mode = False
        self.requests_to_server = {}
        self.requests_lock = threading.Lock()
        self.show_printer_type_selector = False
        self.timeout = self.DEFAULT_OPERATIONAL_TIMEOUT
        self.possible_printer_types = self.get_possible_printer_types()
        self.current_camera = self.parent.camera_controller.get_current_camera_name()
        self.logger.info('New printer interface for %s' % str(usb_info))
        super(PrinterInterface, self).__init__(name="PrinterInterface%s" % self.usb_info)

    def connect_to_server(self):
        self.logger.info("Connecting to server with printer: %s" % str(self.usb_info))
        self.http_client = http_client.HTTPClient(self, keep_connection_flag=True)
        while not self.stop_flag and not self.parent.stop_flag:
            if [error for error in self.errors if error['is_blocking']]:
                self.stop_flag = True
                return False
            message = ['printer_login', self.parent.user_login.user_token, self.usb_info]
            kw_message = {}
            with self.requests_lock:
                if 'select_printer_type' in self.requests_to_server:
                    kw_message['select_printer_type'] = self.requests_to_server['select_printer_type']
                    del self.requests_to_server['select_printer_type']
            self.logger.debug('Printer login request:\n%s\n%s ' % (str(message), str(kw_message)))
            answer = self.http_client.pack_and_send(*message, **kw_message)
            if not answer:
                self.logger.warning("Error on printer login. No connection or answer from server.")
                time.sleep(0.1)
            else:
                error = answer.get('error')
                self.printer_name = answer.get('name', '')
                if error:
                    self.logger.warning("Error while login %s:" % str(self.usb_info))
                    self.logger.warning(str(error['code']) + " " + error["message"])
                    if str(error['code']) == '8':
                        self.show_printer_type_selector = True
                        time.sleep(1)
                        continue
                    else:
                        self.register_error(26, "Server had returned error: %s" % str(error))
                        return False
                elif answer.get('printer_profile') == "":
                    self.register_error(62, "Server return message with empty printer_profile field: %s" % str(error))
                else:
                    groups = answer.get('current_groups', [])
                    self.set_groups(groups)
                    self.show_printer_type_selector = False
                    self.printer_rename_request = False
                    self.logger.info('Successfully connected to server.')
                    self.printer_token = answer['printer_token']
                    self.logger.info('Received answer: ' + str(answer))
                    self.printer_profile = json.loads(answer["printer_profile"])
                    custom_timeout = self.printer_profile.get("operational_timeout", None)
                    if custom_timeout:
                        self.timeout = custom_timeout
                    self.logger.debug('Setting profile: ' + str(self.printer_profile))
                    return True

    def connect_to_printer(self):
        sender_name = self.printer_profile['sender']
        try:
            printer_sender = __import__(sender_name)
        except ImportError:
            message = "Printer type %s not supported by our version of 3DPrinterOS client." % sender_name
            self.register_error(128, message, is_blocking=True)
        else:
            self.logger.info("Connecting with profile: " + str(self.printer_profile))
            try:
                printer = printer_sender.Sender(self, self.usb_info, self.printer_profile)
            except RuntimeError as e:
                message = "Can't connect to printer %s %s\nReason: %s" %\
                          (self.printer_profile['name'], str(self.usb_info), str(e))
                self.register_error(119, message, is_blocking=True)
            else:
                self.logger.info("Successful connection to %s!" % (self.printer_profile['name']))
                return printer

    def form_command_request(self, acknowledge):
        message = [self.printer_token, self.state_report(), acknowledge]
        kw_message = {}
        if self.errors:
            kw_message['error'] = self.errors[:]
            self.errors = []
        self.check_camera_name()
        with self.requests_lock:
            kw_message.update(self.requests_to_server)
            self.requests_to_server = {}
        requests_to_stop_after = ['reset_printer_type']
        for stop_after in requests_to_stop_after:
            if stop_after in kw_message:
                self.stop_in_next_loop_flag = True
                break
        self.logger.debug("Request:\n%s\n%s" % (str(message), str(kw_message)))
        return message, kw_message

    def check_operational_status(self):
        if not self.printer or self.stop_flag:
            pass
        elif self.printer.stop_flag:
            pass
        elif self.printer.is_operational():
            self.last_operational_time = time.time()
            self.forced_state = None
            self.last_error = {}
        elif not self.forced_state:
            message = "Printer is not operational"
            self.register_error(77, message, is_blocking=False)
        elif self.forced_state != "error" and self.last_operational_time + self.timeout < time.time():
            message = "Not operational timeout reached"
            self.register_error(78, message, is_blocking=True)
        for error in self.errors:
            if error['is_blocking']:
                self.forced_state = "error"
                self.close_printer_sender()
                self.stop_in_next_loop_flag = True
            elif not self.forced_state and not error.get('is_info'):
                self.forced_state = "connecting"

    @log.log_exception
    def run(self):
        if self.connect_to_server():
            self.printer = self.connect_to_printer()
            self.last_operational_time = time.time()
            acknowledge = None
            while not self.parent.stop_flag and (not self.stop_flag or self.errors):
                self.check_operational_status()
                message, kw_message = self.form_command_request(acknowledge)
                answer = self.http_client.pack_and_send('command', *message, **kw_message)
                if answer:
                    self.logger.debug("Answer: " + str(answer))
                    acknowledge = self.execute_server_command(answer)
                if not self.stop_flag and not self.parent.stop_flag:
                    if self.stop_in_next_loop_flag:
                        self.stop_flag = True
                    else:
                        time.sleep(self.COMMAND_REQUEST_PERIOD)
            time.sleep(self.COMMAND_REQUEST_PERIOD)
            self.http_client.close()
            self.close_printer_sender()
            self.logger.info('Printer interface was stopped')

    def validate_command(self, server_message):
        command = server_message.get('command')
        if command:
            number = server_message.get('number')
            false_ack = {"number": number, "result": False}
            if self.local_mode:
                self.register_error(111, "Can't execute command %s while in local_mode!" % command, is_blocking = False)
                return false_ack
            if type(number) != int:
                message = "Error in number field of server's message: " + str(server_message)
                self.register_error(41, message,  is_blocking=False)
                return false_ack
            if not hasattr(self, command) and not hasattr(self.printer, command):
                self.register_error(40, "Unknown command:'%s' " % str(command), is_blocking=False)
                return false_ack

    def execute_server_command(self, server_message):
        validation_error = self.validate_command(server_message)
        if validation_error:
            if validation_error['number'] is not None:
                return validation_error
        error = server_message.get('error')
        if error:
            self.logger.warning("@ Server return error: %s\t%s" % (error.get('code'), error.get('message')))
        command, number = server_message.get('command'), server_message.get('number')
        if command:
            # if self.errors:
            #     self.logger.error("! Server returns command on request containing an error - this is error.")
            #     return { "number": number, "result": False }
            self.logger.info("Executing command number %i : %s" % (number, str(command)))
            method = getattr(self, command, None)
            if not method:
                method = getattr(self.printer, command)
            payload = server_message.get('payload')
            if server_message.get('is_link'):
                if self.downloader and self.downloader.is_alive():
                    self.register_error(41, "Can't start new download, because previous download isn't finished.")
                    result = False
                else:
                    if command == 'gcodes':
                        self.printer.set_filename(server_message.get('filename'))
                    self.downloader = downloader.Downloader(
                        self, payload, method, is_zip=bool(server_message.get('zip')),
                        save_raw_gzip=self.printer and self.printer.can_print_gzip
                    )
                    self.downloader.start()
                    result = True
            else:
                if 'payload' in server_message:
                    payload = server_message['payload']
                    if type(payload) == list:
                        arguments = payload
                    else:
                        arguments = [payload]
                else:
                    arguments = []
                try:
                    result = method(*arguments)
                    # to reduce needless 'return True' in methods, we assume that return of None, that is a successful call
                    result = result or result == None
                except Exception as e:
                    message = "! Error while executing command %s, number %d.\t%s" % (command, number, str(e))
                    self.register_error(109, message, is_blocking=False)
                    self.logger.exception(message)
                    result = False
            ack = { "number": number, "result": result }
            return ack

    def get_printer_state(self):
        if self.forced_state:
            state = self.forced_state
        elif (self.printer and self.printer.stop_flag) or not self.printer or self.stop_flag:
            state = 'closing'
        elif self.printer.is_paused():
            state = "paused"
        elif self.downloader and self.downloader.is_alive():
            state = "downloading"
        elif not self.printer.is_bed_clear():
            state = "bed_not_clear"
        elif self.printer.is_printing():
            state = "printing"
        elif self.local_mode:
            state = 'local_mode'
        else:
            state = "ready"
        return state

    def state_report(self):
        report = {"state": self.get_printer_state()}
        if self.printer:
            try:
                if self.downloader and self.downloader.is_alive():
                    report["percent"] = self.downloader.get_percent()
                else:
                    report["percent"] = self.printer.get_percent()
                report["temps"] = self.printer.get_temps()
                report["target_temps"] = self.printer.get_target_temps()
                report["line_number"] = self.printer.get_current_line_number()
                # report["coords"] = self.printer.get_position()
                if self.printer.responses:
                    report["response"] = self.printer.responses[:]
                    self.printer.responses = []
            except Exception as e:
                report["state"] = self.get_printer_state() #update printer state if in was disconnected during report
                self.logger.warning("! Exception while forming printer report: " + str(e))
        return report

    def get_last_error(self):
        try:
            return self.errors_to_display.pop(0)
        except IndexError:
            pass

    def register_error(self, code, message, is_blocking=False, is_info=False):
        error = {"code": code, "message": message, "is_blocking": is_blocking}
        if is_info:
            error['is_info'] = True
        if self.last_error != error:
            self.errors.append(error)
            self.last_error = error
            self.errors_to_display.append(error)
        self.logger.warning("Error N%d. %s" % (code, message))

    def check_camera_name(self):
        camera_name = self.parent.camera_controller.get_current_camera_name()
        if self.current_camera != camera_name:
            self.logger.info("Camera change detected")
            with self.requests_lock:
                self.requests_to_server['camera_change'] = camera_name
            self.current_camera = camera_name

    def get_possible_printer_types(self):
        all_printer_profiles = self.parent.user_login.profiles
        current_printer_vid_pid = [self.usb_info['VID'], self.usb_info['PID']]
        possible_types = [profile for profile in all_printer_profiles if current_printer_vid_pid in profile['vids_pids']]
        possible_types = list(sorted(possible_types, key=lambda pt: pt['name']))
        self.logger.info("Possible printer types: " + str(possible_types))
        return possible_types

    def request_printer_type_selection(self, printer_profile):
        with self.requests_lock:
            self.requests_to_server['select_printer_type'] = printer_profile['alias']
        self.logger.info("Requesting printer type selection: " + printer_profile['alias'])

    def request_printer_groups_selection(self, groups):
        with self.requests_lock:
            self.requests_to_server['selected_groups'] = groups
        self.logger.info("Groups to select: " + str(groups))

    def request_printer_rename(self, name):
        with self.requests_lock:
            self.requests_to_server['select_name'] = name
        self.logger.info("Requesting printer rename: " + str(name))

    def request_reset_printer_type(self):
        with self.requests_lock:
            self.requests_to_server['reset_printer_type'] = True
        self.logger.info("Setting up flag to reset print type in next command request")

    def set_printer_name(self, name):
        self.printer_name = name

    def get_groups(self):
        return self.groups

    def set_groups(self, groups):
        if type(groups) == list:
            self.logger.info("Groups are set correctly for %s %s" % (str(self.usb_info), str(groups)))
            self.groups = groups
        else:
            self.register_error(999, "Invalid workgroups format - type must be list: " + str(groups), is_blocking=False)

    def turn_on_local_mode(self):
        self.local_mode = True

    def turn_off_local_mode(self):
        self.local_mode = False
        try:
            with self.printer.recv_callback_lock:
                self.printer.recv_callback = None
        except:
            pass

    def upload_logs(self):
        self.logger.info("Sending logs")
        if log.report_problem('logs'):
            return False

    def switch_camera(self, module):
        self.logger.info('Changing camera module to %s due to server request' % module)
        self.parent.camera_controller.switch_camera(module, self.printer_token)
        return True

    def restart_camera(self):
        self.logger.info('Executing camera restart command from server')
        self.parent.camera_controller.restart_camera()

    def update_software(self):
        self.logger.info('Executing update command from server')
        self.parent.updater.update()

    def quit_application(self):
        self.logger.info('Received quit command from server!')
        self.parent.stop_flag = True

    def set_name(self, name):
        self.logger.info("Setting printer name: " + str(name))
        self.set_printer_name(name)

    def is_downloading(self):
        return self.downloader and self.downloader.is_alive()

    def cancel(self):
        if self.is_downloading():
            self.logger.info("Canceling downloading")
            self.downloader.cancel()
        else:
            return self.printer.cancel()

    def set_verbose(self, verbose_enabled):
        try:
            if hasattr(self.printer, 'verbose'):
                self.printer.verbose = bool(verbose_enabled)
                self.logger.info("Setting sender verbose to %s" % bool(verbose_enabled))
            if hasattr(self.printer, 'connection') and hasattr(self.printer.connection, 'verbose'):
                self.printer.connection.verbose = bool(verbose_enabled)
                self.logger.info("Setting connection verbose to %s" % bool(verbose_enabled))
        except AttributeError:
            self.register_error(191, "Can't enable runtime verbose - no connection to printer", is_info=True)
            return False
        else:
            return True

    def close_printer_sender(self):
        if self.printer and not self.printer.stop_flag:
            self.logger.info('Closing ' + str(self.printer_profile))
            self.printer.close()
            self.logger.info('...closed.')

    def close(self):
        self.logger.info('Closing printer interface: ' + str(self.usb_info))
        self.stop_flag = True

    def report_problem(self, problem_description):
        log.report_problem(problem_description)
