#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import json
import paths
import logging
import marshal

import config
import http_client
from checker_waiter import CheckerWaiter

storage_path = os.path.join(paths.CURRENT_SETTINGS_FOLDER, 'stuff.bin')

def read_login():
    logger = logging.getLogger(__name__)
    try:
        with open(storage_path, "rb") as f:
            marshalled_data = f.read()
    except (IOError, OSError) as e:
        logger.info("Can't read login storage file: " + str(e))
    else:
        try:
            login_password = marshal.loads(marshalled_data)
            for i in range(len(login_password)):
                if isinstance(login_password[i], bytes):
                    login_password[i] = login_password[i].decode('utf-8')
            if len(login_password) < 3:
                login_password.append("")

            # print('logindata')
            # for x in login_password:
            #     print(x, end="\n")
        except:
            logger.info("Can't decode login and password")
        else:
            return login_password

#login storage with some rational paranoia
def write_login(login, password, host_access_token=None):
    logger = logging.getLogger(__name__)
    marshalled_data = marshal.dumps([login, password, host_access_token])
    new_storage_file = os.path.join(paths.CURRENT_SETTINGS_FOLDER, 'new_stuff.bin')
    try:
        with open(new_storage_file, "wb") as f:
            f.write(marshalled_data)
    except (IOError, OSError) as e:
        logger.warning('Error writing login to storage file: ' + str(e))
    else:
        try:
            os.remove(storage_path)
        except OSError:
            pass
        try:
            os.rename(new_storage_file, storage_path)
            return True
        except (IOError, OSError) as e:
            logger.warning('Error renaming login to storage file: ' + str(e))
        try:
            os.rename(new_storage_file, storage_path)
            return True
        except (IOError, OSError) as e:
            logger.warning('Error renaming login to storage file even after removal of some logs: ' + str(e))

def logout():
    logger = logging.getLogger(__name__)
    logger.info("Removing login file")
    try:
        os.remove(storage_path)
    except:
        logger.info("Login file removed")

def apply_settings_mod(settings_mod):
    current_settings = config.get_settings()
    new_settings = config.merge_dictionaries(current_settings, settings_mod, overwrite=True)
    config.Config.instance().settings = new_settings
    logger = logging.getLogger(__name__)
    logger.info("Setting modifications:\n" + str(new_settings))


class UserLogin(CheckerWaiter):

    NAME_FOR_LOGGING = 'User login'

    def __init__(self, app, auto_login=True):
        self.user_token = None
        CheckerWaiter.__init__(self, app, lambda: bool(self.user_token))
        self.app = app
        self.login = None
        self.mac = ''
        if auto_login:
            login_password = read_login()
            if login_password:
                error = self.login_as_user(login_password[0], login_password[1],
                                           host_access_token=login_password[2],
                                           save_password_flag=False, save_host_access_token_flag=True)
                if error:
                    self.logger.info(str(error))

    def login_as_user(self, login=None, password=None, disposable_token=None,
                      host_access_token=None, save_password_flag=True,
                      save_host_access_token_flag=False):
        http_connection = http_client.HTTPClient(self.app)
        if not login:
            return 0, "Empty login"
        if password is None and not disposable_token and not host_access_token:
            return 0, "Empty password"
        answer = http_connection.pack_and_send('user_login',
                                               login, password, disposable_token=disposable_token,
                                               host_access_token=host_access_token)
        if answer:
            user_token = answer.get('user_token')
            profiles_str = answer.get('all_profiles')
            settings_mod = answer.get('settings_mod')
            if settings_mod:
                apply_settings_mod(settings_mod)
            error = answer.get('error', None)
            if error:
                self.logger.warning("Error processing user_login " + str(error))
                self.logger.error("Login rejected")
                return error['code'], error['message']
            self.user_token = user_token
            if login and (save_password_flag or save_host_access_token_flag):
                if config.get_settings()['login_host_access_token'] and answer.get('host_access_token'):
                    if answer.get('host_access_token') != host_access_token or password is not None:
                        write_login(login, None, answer.get('host_access_token'))
                elif save_password_flag:
                    write_login(login, password)
            try:
                profiles = json.loads(profiles_str)
                if not profiles:
                    raise RuntimeError("Server returned empty profiles on user login")
            except Exception as e:
                self.logger.warning("Error while parsing profiles: " + str(e))
                return 42, "Error parsing profiles"
            else:
                self.profiles = profiles
                self.mac = http_connection.mac
                if login:
                    self.login = login
                else:
                    login_name = answer.get('user_login')
                    if login_name:
                        self.login = login_name
                    else:
                        self.login = "Temporary login"
                self.logger.info("Successful login from user %s" % self.login)
