#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import re
import sys
import time
import json
import uuid
import http.client
import logging
import ssl
from subprocess import Popen, PIPE

import config
import version
import platforms

class HTTPClient:

    URL = config.get_settings()['URL']
    HTTPS_MODE = config.get_settings()['HTTPS']
    ALLOW_UNVERIFIED_SERVER = config.get_settings()['allow_unverified_server']
    BASE_TIMEOUT = 6
    MAX_TIMEOUT = BASE_TIMEOUT * 5

    streamer_prefix = "/streamerapi"
    user_login_path = streamer_prefix + "/user_login"
    printer_login_path = streamer_prefix + "/printer_login"
    command_path = streamer_prefix + "/command"
    token_send_logs_path = streamer_prefix + "/sendLogs"
    camera_path = streamer_prefix + "/camera" #json['image': base64_image ]

    def __init__(self, parent, keep_connection_flag = False, debug = False):
        self.logger = logging.getLogger(__name__)
        self.parent = parent
        self.parent_usb_info = getattr(parent, 'usb_info', None)
        if debug:
            self.logger.setLevel('DEBUG')
        else:
            self.logger.setLevel('INFO')
        self.keep_connection_flag = keep_connection_flag
        self.timeout = self.BASE_TIMEOUT
        if hasattr(parent, "parent"): #TODO refactor mess with non universal mac and local_ip
            app = parent.parent
        else:
            app = parent
        self.mac = getattr(getattr(app, "user_login", None), "mac", None)
        self.local_ip = None
        self.connection = self.connect()

    def get_mac_add_or_serial_number(self):
        id = self.get_serial_number()
        if not id:
            id = self.get_mac_for_current_ip()
        if not id:
            time.sleep(1)
            id = self.get_mac_for_current_ip()
        if not id:
            self.logger.error("Warning! Can't get mac address! Using uuid.getnode().")
            id = hex(uuid.getnode())
        return id

    # machine id is mac address, but on RPi we use machine serial
    @staticmethod
    def get_serial_number():
        if sys.platform.startswith('linux'):
            call = Popen(['cat', '/proc/cpuinfo'], stdout=PIPE, stderr=PIPE)
            stdout, stderr = call.communicate()
            stdout = stdout.decode('utf-8').replace('\t', '').split('\n')
            for item in stdout:
                if 'Serial: ' in item:
                    serial = item.replace('Serial: ', '').strip()
                    if serial and serial == "0" * len(serial):
                        return None
                    return serial

    @staticmethod
    def format_mac_addr(mac):
        mac = mac.replace(":", "")
        mac = mac.replace("-", "")
        mac = mac.lower()
        return "0x" + mac + "L"

    def get_mac_for_current_ip(self):
        if platforms.PLATFORM in ("rpi", 'linux', "mac"):
            process = Popen(['ifconfig'], stdout=PIPE)
            stdout, _ = process.communicate()
            stdout = stdout.decode('utf-8')
            for splitter in ("flags=", "Link"):
                if splitter in stdout:
                    interfaces = stdout.split(splitter) #can get name of interface wrong, but we don't need name
                    break
            else:
                return False
            for interface in interfaces:
                if "inet " + self.local_ip in interface:
                    search = re.search("ether\s([0-9a-f\:]+)", interface)
                    if search:
                        return self.format_mac_addr(search.group(1))
                elif "inet addr:" + self.local_ip in interface:
                        search = re.search("HWaddr\s([0-9a-fA-F\:]+)", interface)
                        if search:
                            return self.format_mac_addr(search.group(1))
        else:
            process = Popen('ipconfig /all', stdout=PIPE, shell=True)
            stdout, _ = process.communicate()
            interfaces = stdout.split("\r\n\r\n")
            for interface in interfaces:
                search = re.search("IP.*:\s(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})", interface)
                if search:
                    ip = search.group(1)
                    if ip == self.local_ip:
                        search = re.search("[A-F0-9]{2}\-[A-F0-9]{2}\-[A-F0-9]{2}\-[A-F0-9]{2}\-[A-F0-9]{2}\-[A-F0-9]{2}", interface)
                        if search:
                            return self.format_mac_addr(search.group(0))

    def connect(self):
        self.logger.debug("{ Connecting...")
        while not self.parent.stop_flag:
            try:
                if self.HTTPS_MODE:
                    if self.ALLOW_UNVERIFIED_SERVER:
                        context = ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
                        context.verify_mode = False
                        connection = http.client.HTTPSConnection(self.URL, port = 443, timeout = self.timeout,
                                                                 context=context)
                    else:
                        connection = http.client.HTTPSConnection(self.URL, port = 443, timeout = self.timeout)
                else:
                    connection = http.client.HTTPConnection(self.URL, port = 80, timeout = self.timeout)
                connection.connect()
                self.local_ip = connection.sock.getsockname()[0]
                if not self.mac:
                    self.mac = self.get_mac_add_or_serial_number()
            except Exception as e:
                self.parent.register_error(5, "Error during HTTP connection: " + str(e))
                self.logger.debug("...failed }")
                self.logger.warning("Warning: connection to %s failed." % self.URL)
                if self.timeout < self.MAX_TIMEOUT:
                    self.timeout += self.BASE_TIMEOUT
                time.sleep(1)
            else:
                self.logger.debug("...success }")
                self.logger.info("Connecting from: %s\t%s" % (self.local_ip, self.mac))
                return connection

    def load_json(self, jdata):
        try:
            data = json.loads(jdata)
        except ValueError:
            self.parent.register_error(2, "Received data is not valid json: " + jdata)
        else:
            if data and type(data) == dict:
                return data
            else:
                message = "Error parsing http message, data should be not empty dictionary: " + str(data)
                self.parent.register_error(3, message)

    def request(self, method, connection, path, payload, headers=None):
        self.logger.debug("{ Requesting...")
        if headers is None:
            headers = {"Content-Type": "application/json", "Content-Length": str(len(payload))}
            if self.keep_connection_flag:
                headers["Connection"] = "keep-alive"
        try:
            connection.request(method, path, payload, headers)
            resp = connection.getresponse()
        except Exception as e:
            self.parent.register_error(6, "Error during HTTP request:" + str(e), is_info=True)
            time.sleep(1)
        else:
            #self.logger.debug("Request status: %s %s" % (resp.status, resp.reason))
            try:
                received = resp.read()
            except Exception as e:
                self.parent.register_error(7, "Error reading response: " + str(e), is_info=True)
            else:
                if resp.status == http.client.OK and resp.reason == "OK":
                    self.logger.debug("...success }")
                    return received
                else:
                    message = "Error: server responded with non 200 OK\nFull response: %s" % received
                    self.parent.register_error(8, message, is_info=True)
        self.logger.debug("...failed }")
        self.logger.warning("Warning: HTTP request failed!")

    def pack_and_send(self, target, *payloads, **kwargs_payloads):
        path, packed_message = self.pack(target, *payloads, **kwargs_payloads)
        return self.send(path, packed_message)

    def send(self, path, data):
        while not self.parent.stop_flag:
            if not self.connection:
                self.connection = self.connect()
            answer = self.request("POST", self.connection, path, data)
            if not self.keep_connection_flag or not answer:
                self.connection.close()
                self.connection = None
            if answer:
                return self.load_json(answer.decode('utf-8'))
            # Some delay before next request
            time.sleep(3)

    def pack(self, target, *args, **kwargs):
        if target == 'user_login':
            data = { 'login': {'user': args[0], 'password': args[1]},
                     'platform': platforms.PLATFORM, 'host_mac': self.mac,
                     'local_ip': self.local_ip, 'version': version.version + version.branch}
            if 'disposable_token' in kwargs:
                data['login']['disposable_token'] = kwargs['disposable_token']
                del kwargs['disposable_token']
            if 'host_access_token' in kwargs:
                data['login']['host_access_token'] = kwargs['host_access_token']
                del kwargs['host_access_token']
            path = self.user_login_path
            #self.logger.debug(data)
        elif target == 'printer_login':
            data = { 'user_token': args[0], 'printer': args[1], "version": version.version + version.branch,
                     "data_time": time.ctime(), "camera": config.get_app().camera_controller.get_current_camera_name(),
                     "camera_number_on_creating": 1, 'verbose': config.get_settings()['verbose']}
            path = self.printer_login_path
        elif target == 'command':
            data = { 'printer_token': args[0], 'report': args[1], 'command_ack': args[2] }
            if not data['command_ack']:
                data.pop('command_ack')
            path = self.command_path
        elif target == 'camera':
            data = { 'user_token': args[0], 'camera_number': args[1], 'camera_name': args[2],
                     'file_data': args[3], 'host_mac': self.mac}
            path = self.camera_path
        else:
            self.parent.register_error(4, 'No such target for packaging: ' + target)
            data, path = None, None
        for key, value in list(kwargs.items()):
            data[key] = value
        # self.logger.info(data)
        return path, json.dumps(data)

    def close(self):
        if self.connection:
            self.connection.close()
