#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import sys

import platform

LIBRARIES_FOLDER = 'libraries'
LIBRARIES = ['opencv', 'numpy', 'wx', 'rumps']

def check_and_create_dirs(path):
    while path:
        path = os.path.dirname(path)
        if os.path.exists(path):
            break
        else:
            os.mkdir(path)

def init_path_to_libs():
    #logger = logging.getLogger(__name__)
    if sys.platform.startswith('win'):
        folder_name = "win"
        LIBRARIES.append('pywin')
    elif sys.platform.startswith('linux'):
        folder_name = "linux"
    elif sys.platform.startswith('darwin'):
        folder_name = "mac"
    else:
        raise RuntimeError('Cannot define operating system')
    our_dir = os.path.dirname(os.path.abspath(__file__))
    platform_dir = os.path.join(our_dir, LIBRARIES_FOLDER, folder_name)
    for lib in LIBRARIES:
        lib_path = os.path.join(platform_dir, lib)
        #logger.info('Using library: ' + lib_path)
        #print 'Using library: ' + lib_path
        sys.path.append(lib_path)

def get_libusb_path(lib):
    #logger = logging.getLogger(__name__)
    #logger.debug('Using: ' + lib)
    if sys.platform.startswith('win'):
        folder_name = "win"
        python_version = platform.architecture()[0]
        if '64' in python_version:
            libusb_name = 'libusb-1.0-64.dll'
        else:
            libusb_name = 'libusb-1.0.dll'
    elif sys.platform.startswith('linux'):
        folder_name = "linux"
        libusb_name = 'libusb-1.0.so'
    elif sys.platform.startswith('darwin'):
        folder_name = "mac"
        libusb_name = 'libusb-1.0.dylib'
    else:
        raise EnvironmentError('Could not detect OS. Only GNU/LINUX, MAC OS X and MS WIN VISTA/7/8 are supported.')
    our_dir = os.path.dirname(os.path.abspath(__file__))
    backend_path = os.path.join(our_dir, LIBRARIES_FOLDER, folder_name, 'libusb', libusb_name)
    #logger.debug('Libusb from: ' + backend_path)
    return backend_path

def get_paths_to_settings_folder():
    abs_path_to_users_home = os.path.abspath(os.path.expanduser("~"))
    folder_name = '.3dprinteros'
    if sys.platform.startswith('win'):
        abs_path_to_appdata = os.path.abspath(os.getenv('APPDATA'))
        path = os.path.join(abs_path_to_appdata, '3dprinteros')
    elif sys.platform.startswith('linux') or sys.platform.startswith('darwin'):
        path = os.path.join(abs_path_to_users_home, folder_name)
    else:
        raise EnvironmentError('Could not detect OS. Only GNU/LINUX, MAC OS X and MS WIN VISTA/7/8 are supported.')
    local_path = os.path.dirname(os.path.abspath(__file__))
    if not os.path.exists(path):
        os.mkdir(path)
    return (path, local_path)

def current_settings_folder():
    return get_paths_to_settings_folder()[0]

CURRENT_SETTINGS_FOLDER = current_settings_folder()
UPDATE_FILE_PATH = os.path.join(CURRENT_SETTINGS_FOLDER, '3dprinteros_client_update.zip')