#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import os
import logging
import urllib.request, urllib.parse, urllib.error
import time
import json
import zipfile
import http.client
import threading
import ssl

import version
import config
import platforms
import paths


class Updater:

    IDLE_TIMEOUT_FOR_AUTOUPDATE = 120
    ALLOW_UNVERIFIED_SERVER = config.get_settings()['allow_unverified_server']
    ALLOW_UNVERIFIED_STORAGE = config.get_settings()['allow_unverified_storage']

    def __init__(self, parent):
        self.logger = logging.getLogger(__name__)
        self.update_download_url = None
        self.update_available = False
        self.check_time = 0
        self.updating_thread = None
        self.parent = parent
        self.unverified_ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
        self.unverified_ssl_context.verify_mode = False
        if config.get_settings()['update'].get('no_url_prefix', False):
            self.host = config.get_settings()['URL']
        else:
            try:
                self.host = "%s-update.%s.%s" % tuple(config.get_settings()['URL'].split('.'))
            except:
                self.logger.info("Strange update URL detected. Falling back to base URL")
                self.host = config.get_settings()['URL']

    def check_update_timer(self):
        current_time = time.time()
        if current_time - self.check_time > config.get_settings()['update']['check_pause']:
            self.check_time = current_time
            self.check_for_updates()

    def check_for_updates(self, force=False):
        if config.get_settings()['update']['enabled']:
            self.update_download_url = self.request_update_download_url()
            if self.update_download_url:
                self.logger.info('Updates available!')
                self.update_available = True
                if force or config.get_settings()['update']['auto_update_enabled']:
                    if not (self.updating_thread and self.updating_thread.is_alive()):
                        self.updating_thread = threading.Thread(target = self.auto_update)
                        self.updating_thread.start()

    def request_update_download_url(self):
        payload = json.dumps( { 'platform': platforms.get_platform(), 'client_version': version.version,
                                'branch': version.branch, 'build': version.build } )
        headers = {"Content-Type": "application/json", "Content-Length": str(len(payload))}
        try:
            if config.get_settings()['HTTPS']:
                if self.ALLOW_UNVERIFIED_SERVER:
                    connection = http.client.HTTPSConnection(self.host, timeout=2, context=self.unverified_ssl_context)
                else:
                    connection = http.client.HTTPSConnection(self.host, timeout=2)
            else:
                connection = http.client.HTTPConnection(self.host, timeout=2)
            connection.connect()
            connection.request('POST', '/noauth/get_client_update_url', payload, headers)
            resp = connection.getresponse()
            response = resp.read().decode('utf-8')
            if resp.getcode() == 200:
                update_download_url = response
            else:
                msg = "Some error on updater server: code="+str(resp.getcode())+" response="+response
                self.logger.warning(msg)
                update_download_url = ''
            connection.close()
        except Exception as e:
            self.logger.warning('Unable to connect to updater server. ' + str(e))
        else:
            if update_download_url:
                self.logger.debug('Update address received: ' + update_download_url)
                return update_download_url

    def auto_update(self):
        self.logger.info('Starting automatic update')
        error = self.download_update()
        if not error:
            self.wait_for_prints_end()

    def wait_for_prints_end(self):
        self.logger.info('Waiting for prints to end to install update...')
        last_non_idle_time = time.time()
        while not self.parent.stop_flag:
            for printer_interface in config.get_app().printer_interfaces:
                if printer_interface.get_printer_state() != 'ready':
                    last_non_idle_time = time.time()
            if time.time() - last_non_idle_time > self.IDLE_TIMEOUT_FOR_AUTOUPDATE:
                self.logger.info('...all prints are finished. Rebooting client...')
                config.get_app().stop_flag = True
            time.sleep(3)

    def download_update(self):
        if self.update_available:
            if os.path.exists(paths.UPDATE_FILE_PATH) and zipfile.is_zipfile(paths.UPDATE_FILE_PATH):
                self.logger.info('Update package already downloaded')
            else:
                self.logger.info('Downloading update package...')
                try:
                    if self.ALLOW_UNVERIFIED_STORAGE:
                        with urllib.request.urlopen(self.update_download_url, context=self.unverified_ssl_context) as u:
                            with open(paths.UPDATE_FILE_PATH, 'wb') as f:
                                f.write(u.read(1024*1024))
                    else:
                        with urllib.request.urlopen(self.update_download_url) as u:
                            with open(paths.UPDATE_FILE_PATH, 'wb') as f:
                                f.write(u.read(1024*1024))
                except Exception as e:
                    error_message = 'Update download failed: error while downloading from %s\nDetails: %s' %\
                                        ( self.update_download_url, str(e) )
                    self.logger.warning(error_message)
                    return error_message
                else:
                    if zipfile.is_zipfile(paths.UPDATE_FILE_PATH):
                        self.logger.info('...update successfully downloaded!')
                        self.update_available = False
                    else:
                        error_message = 'Error: corrupted update package.'
                        self.logger.warning(error_message)
                        try:
                            os.remove(paths.UPDATE_FILE_PATH)
                        except:
                            self.logger.warning("Unable to remove corrupted update package.")
                        else:
                            self.logger.warning("Corrupted update package removed.")
                        return error_message

    def update(self):
        self.check_for_updates(force=True)


if __name__ == "__main__":

    class FakeApp:
        def __init__(self):
            self.stop_flag = False

    fa = FakeApp()
    u = Updater(fa)
    print(u.request_update_download_url())
