#
# Copyright 3D Control Systems, Inc. All Rights Reserved 2017-2019. Built in San Francisco.
#
# This software is distributed under commercial non-GPL license for personal, educational,
# corporate or any other use. The software as a whole or any parts of that are prohibited
# for distribution and/or use without obtaining license from 3D Control Systems, Inc.
#
# If you do not have the license to use this software, please delete all software files
# immediately and contact sales to obtain the license: sales@3dprinteros.com.
# If you are unsure about the licensing please contact directly our sales: sales@3dprinteros.com.

import time
import threading

from base_sender import BaseSender

class Sender(BaseSender):

    def __init__(self, parent, usb_info, profile):
        BaseSender.__init__(self, parent, usb_info, profile)
        self.print_thread = None
        self.printing_flag = False
        self.total_gcodes = 0
        self.lines = 0
        self.pause_flag = False

    def is_operational(self):
        return not self.stop_flag

    def is_printing(self):
        return self.printing_flag

    def is_paused(self):
        return self.pause_flag

    def load_gcodes(self, gcodes):
        self.logger.info("Virtual printer loading gcodes")
        self.lines = len(self.preprocess_gcodes(gcodes))
        self.total_gcodes = self.lines
        self.print_thread = threading.Thread(target=self.printing)
        self.print_thread.start()

    def unbuffered_gcodes(self, gcodes):
        self.logger.info("Virtual printer unbuffered gcodes: %s" % gcodes)

    def printing(self):
        self.printing_flag = True
        while self.lines > 0 and not self.stop_flag:
            if not self.pause_flag:
                self.lines -= 10
            time.sleep(1)
        self.printing_flag = False

    def cancel(self):
        self.logger.info("Virtual printer cancel")
        self.lines = 0

    def pause(self):
        self.logger.info("Virtual printer pause")
        self.pause_flag = True

    def unpause(self):
        self.logger.info("Virtual printer unpause")
        self.pause_flag = False

    def get_percent(self):
        if self.total_gcodes:
            return int(self.get_current_line_number() / float(self.total_gcodes) * 100)
        else:
            return 0

    def get_current_line_number(self):
        return self.total_gcodes - self.lines

    def reset(self):
        self.logger.info("Virtual printer reset")

    def close(self):
        self.logger.info('Closing Virtual printer sender')
        self.stop_flag = True
        if self.print_thread:
            self.print_thread.join()


